<# :# PowerShell comment protecting the Batch section
@echo off
:# Written by Boris Fuxon aka Bifido (bifido.net) .
:# This code may be freely used, modified, and distributed.
:# However it is provided with no warranty, either express or implied.
color 07
:# Disabling argument expansion avoids issues with ! in arguments (EnableDelayedExpansion).
SetLocal EnableExtensions
:# The script checks if bat file executed with Admin priveleges
call :Admin

:# ।  ப 㦭 ࠧ來.
if exist %SystemRoot%\sysnative\cmd.exe (
	%SystemRoot%\sysnative\cmd.exe /c "%~f0" %*
	goto :eof
)

:# Read about different variants of running a PowerShell script within a Windows batch file in this articles
:# https://stackoverflow.com/questions/2609985/how-to-run-a-powershell-script-within-a-windows-batch-file
:# https://www.dostips.com/forum/viewtopic.php?f=3&t=5526
:# https://www.dostips.com/forum/viewtopic.php?p=37780

:# Prepare the batch arguments, so that PowerShell parses them correctly
set POWERSHELL_BAT_ARGS=%*
if defined POWERSHELL_BAT_ARGS set POWERSHELL_BAT_ARGS=%POWERSHELL_BAT_ARGS:"=\"%
if defined POWERSHELL_BAT_ARGS set POWERSHELL_BAT_ARGS=%POWERSHELL_BAT_ARGS:'=''%

:# The ^ before the first " ensures that the Batch parser does not enter quoted mode
:# there, but that it enters and exits quoted mode for every subsequent pair of ".
:# This in turn protects the possible special chars & | < > within quoted arguments.
:# Then the \ before each pair of " ensures that PowerShell's C command line parser 
:# considers these pairs as part of the first and only argument following -c.
:# Cherry on the cake, it's possible to pass a " to PS by entering two "" in the bat args.
powershell -NoLogo -NoProfile -Command ^"Invoke-Expression ('^& {' + [io.file]::ReadAllText(\"%~f0\") + '} %POWERSHELL_BAT_ARGS%')"

:# Some other variants
:# powershell -NoLogo -NoProfile -Command "$input | &{ [ScriptBlock]::Create( ( Get-Content \"%~f0\" ) -join [char]10 ).Invoke( @( &{ $args } %POWERSHELL_BAT_ARGS% ) ) }"
:# powershell -NoLogo -NoProfile -Command "$_ = $input; Invoke-Expression $( '$input = $_; $_ = \"\"; $args = @( &{ $args } %POWERSHELL_BAT_ARGS% );' + [String]::Join( [char]10, $( Get-Content \"%~f0\" ) ) )"
:# powershell -NoLogo -NoProfile -Command "Invoke-Expression (${"%~f0"} | out-string)"

if not %errorlevel% EQU 0 (timeout /t 60)
goto :EOF

:# --- The script checks if bat file executed with Admin priveleges ---
:Admin
reg query "HKU\S-1-5-19\Environment" >nul 2>&1
if not %errorlevel% EQU 0 (
	cls
	:# cmd /u /c mshta vbscript:Execute^("CreateObject(""Shell.Application"").ShellExecute ""%~f0"",""%1 %2"","""",""runas"",1:close"^)
	:# powershell.exe -windowstyle hidden -noprofile "$shell = New-Object -ComObject Shell.Application;$shell.ShellExecute('%~s0', '', '', 'runas')"
	powershell.exe -windowstyle hidden -noprofile "Start-Process '%~dpnx0' -Verb RunAs"
	exit
)
exit /b
###############################################################################
End of the PS comment around the Batch section; Begin the PowerShell section #>

function UpdateTools {
	[CmdletBinding(
		ConfirmImpact = 'Medium',
		SupportsShouldProcess = $true
	)]
	param();
	# ᨭ 㬥⮢
	$args | %{ "PowerShell args[{0}] = '$_'" -f $i++ }

	
	function RunAsTrustedInstaller
	{
		# Spawn a process under a different parent process 
		Set-StrictMode -Version Latest
		# Definition is required to calculate hash sum on powershell 2
		Add-Type -AssemblyName System.Core
		# Definition for WinApi functions
		Add-Type -TypeDefinition @"
		using System;
		using System.Diagnostics;
		using System.IO;
		using System.Runtime.InteropServices;

			public class Win32API
			{
				[DllImport("kernel32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool CreateProcess(
				string lpApplicationName, string lpCommandLine, ref SECURITY_ATTRIBUTES lpProcessAttributes,
				ref SECURITY_ATTRIBUTES lpThreadAttributes, bool bInheritHandles, uint dwCreationFlags,
				IntPtr lpEnvironment, string lpCurrentDirectory, ref STARTUPINFOEX lpStartupInfo,
				out PROCESS_INFORMATION lpProcessInformation);

				[DllImport("kernel32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool UpdateProcThreadAttribute(
				IntPtr lpAttributeList, uint dwFlags, IntPtr Attribute, IntPtr lpValue,
				IntPtr cbSize, IntPtr lpPreviousValue, IntPtr lpReturnSize);

				[DllImport("kernel32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool InitializeProcThreadAttributeList(
				IntPtr lpAttributeList, int dwAttributeCount, int dwFlags, ref IntPtr lpSize);

				[DllImport("kernel32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool DeleteProcThreadAttributeList(IntPtr lpAttributeList);

				[DllImport("kernel32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool CloseHandle(IntPtr hObject);
				
				[DllImport("kernel32.dll", SetLastError = true)]
				public static extern IntPtr GetConsoleWindow();
				
				[DllImport("user32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);
				
				
				[DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool AdjustTokenPrivileges(
				IntPtr htok, bool disall, ref TokPriv1Luid newst, int len, IntPtr prev, IntPtr rele);

				[DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool OpenProcessToken(IntPtr h, int acc, ref IntPtr phtok);

				[DllImport("advapi32.dll", SetLastError = true)]
				[return: MarshalAs(UnmanagedType.Bool)]
				public static extern bool LookupPrivilegeValue(string host, string name, ref long pluid);
					
				[DllImport("kernel32.dll", CharSet = CharSet.Auto, SetLastError = true)]
				public static extern IntPtr GetCurrentProcess();

			}
			
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
			public struct STARTUPINFO
			{
				public Int32 cb;
				public IntPtr lpReserved;
				public IntPtr lpDesktop;
				public IntPtr lpTitle;
				public Int32 dwX;
				public Int32 dwY;
				public Int32 dwXSize;
				public Int32 dwYSize;
				public Int32 dwXCountChars;
				public Int32 dwYCountChars;
				public Int32 dwFillAttribute;
				public Int32 dwFlags;
				public Int16 wShowWindow;
				public Int16 cbReserved2;
				public IntPtr lpReserved2;
				public IntPtr hStdInput;
				public IntPtr hStdOutput;
				public IntPtr hStdError;
			}

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
			public struct STARTUPINFOEX
			{
				public STARTUPINFO startupInfo;
				public IntPtr lpAttributeList;
			}

			[StructLayout(LayoutKind.Sequential)]
			public struct PROCESS_INFORMATION
			{
				public IntPtr hProcess;
				public IntPtr hThread;
				public int dwProcessId;
				public int dwThreadId;
			}

			[StructLayout(LayoutKind.Sequential)]
			public struct SECURITY_ATTRIBUTES
			{
				public int nLength;
				public IntPtr lpSecurityDescriptor;
				public int bInheritHandle;
			}
			
			[StructLayout(LayoutKind.Sequential, Pack = 1)]
			public struct TokPriv1Luid {
				public int Count;
				public long Luid;
				public int Attr;
			}

"@
		
		Function CreateProcessFromParent([UInt32]$processPID, [String]$lpApplicationName, [String]$lpCommandLine)
		{
			# First investigation of possibilities for the method by Didier Stevens - 
			# https://blog.didierstevens.com/2009/11/22/quickpost-selectmyparent-or-playing-with-the-windows-process-tree/
			# And practical detailed explanation the logic of this method by James Forshaw - 
			# https://tyranidslair.blogspot.com/2017/08/the-art-of-becoming-trustedinstaller.html
			
			$Win32Constants = New-Object PSObject -Property @{
				EXTENDED_STARTUPINFO_PRESENT = 0x00080000;
				CREATE_NEW_CONSOLE = 0x00000010;
				PROC_THREAD_ATTRIBUTE_PARENT_PROCESS = 0x00020000;
				CREATE_UNICODE_ENVIRONMENT = 0x00000400;
				STARTF_USECOUNTCHARS = 0x00000008;
			}
			
			[IntPtr] $lpValue = [IntPtr]::Zero
			# Creating the STARTUPINFOEX structure
			[STARTUPINFOEX]$si = New-Object STARTUPINFOEX -Property @{
				# Creating the STARTUPINFO structure
				startupInfo = New-Object STARTUPINFO -Property @{
					# The size of structure is the size of STARTUPINFOEX structure
					cb = [System.Runtime.InteropServices.Marshal]::SizeOf([Type][STARTUPINFOEX])
					# Specifies the screen buffer height
					dwFlags = ($Win32Constants.STARTF_USECOUNTCHARS)
					dwYCountChars = 9000
				}
				lpAttributeList = [IntPtr]::Zero
			}
			# Creating the PROCESS_INFORMATION structure
			[PROCESS_INFORMATION]$pi = New-Object PROCESS_INFORMATION
			# Creating the SECURITY_ATTRIBUTES structure
			[SECURITY_ATTRIBUTES]$sa = New-Object SECURITY_ATTRIBUTES -Property @{
				nLength = [System.Runtime.InteropServices.Marshal]::SizeOf([Type][SECURITY_ATTRIBUTES])
			}	
			
			try
			{
				# Enable the seDebugPrivilege
				[System.Diagnostics.Process]::EnterDebugMode()
				$Result = $true
				Switch ($true){
					($Result) {
						# First call this function return an error, but we get the right size for allocate the memory
						[IntPtr]$lpSize = [IntPtr]::Zero
						$Result = [Win32API]::InitializeProcThreadAttributeList([IntPtr]::Zero, 1, 0, [ref]$lpSize)
						# ERROR_INSUFFICIENT_BUFFER
						If([System.Runtime.InteropServices.Marshal]::GetLastWin32Error() -eq 122) {	$Result = $true	}
					}
					($Result) {
						# Allocate corresponding amount of memory for process attributes in STARTUPINFOEX structure
						$si.lpAttributeList = [System.Runtime.InteropServices.Marshal]::AllocHGlobal($lpSize)
						# Initializing this attributes
						$Result = [Win32API]::InitializeProcThreadAttributeList($si.lpAttributeList, 1, 0, [ref]$lpSize)
					}
					($Result) {
						# Allocate unmanaged memory for the pointer
						$lpValue = [System.Runtime.InteropServices.Marshal]::AllocHGlobal([System.Runtime.InteropServices.Marshal]::SizeOf([Type][IntPtr]))
						# Writing in the pointer the value of parent handle address
						[System.Runtime.InteropServices.Marshal]::WriteIntPtr($lpValue, [System.Diagnostics.Process]::GetProcessById($processPID).Handle)
						# Fill this reserved attributes as different parent process entries
						$Result = [Win32API]::UpdateProcThreadAttribute($si.lpAttributeList, 0, $Win32Constants.PROC_THREAD_ATTRIBUTE_PARENT_PROCESS, $lpValue, [IntPtr]([IntPtr]::Size), [IntPtr]::Zero, [IntPtr]::Zero)
					}
					($Result) {
						# Create the child process under a different parent process with this attributes
						$Result = [Win32API]::CreateProcess($lpApplicationName, $lpCommandLine, [ref]$sa, [ref]$sa, $false, ($Win32Constants.EXTENDED_STARTUPINFO_PRESENT -bor $Win32Constants.CREATE_UNICODE_ENVIRONMENT -bor $Win32Constants.CREATE_NEW_CONSOLE), [IntPtr]::Zero, "$env:systemroot\System32", [ref]$si, [ref]$pi)
					}
				}
				return $Result
				# Error codes - https://docs.microsoft.com/en-us/windows/desktop/Debug/system-error-codes
			}
			finally
			{
				Switch ($true){
					($si.lpAttributeList -ne [IntPtr]::Zero) {
						# Free the process attributes
						[void][Win32API]::DeleteProcThreadAttributeList($si.lpAttributeList)
						# Free the unmanaged allocated memory
						[System.Runtime.InteropServices.Marshal]::FreeHGlobal($si.lpAttributeList)
					}
					($lpValue -ne [IntPtr]::Zero) {
						# Free the unmanaged allocated memory
						[System.Runtime.InteropServices.Marshal]::FreeHGlobal($lpValue)
					}
					#Free the handles returned in the ProcessInfo structure
					($pi.hProcess -ne [IntPtr]::Zero) {
						If(![Win32API]::CloseHandle($pi.hProcess)) {
							Write-Warning "Closing process handle failed, Win32Error: $([System.Runtime.InteropServices.Marshal]::GetLastWin32Error())"
						}
					}
					($pi.hThread -ne [IntPtr]::Zero) {
						If(![Win32API]::CloseHandle($pi.hThread)) {
							Write-Warning "Closing thread handle failed, Win32Error: $([System.Runtime.InteropServices.Marshal]::GetLastWin32Error())"
						}
					}
				}
			}
		}
		
		function Elevate-Privileges ([string] $Privilege) {
		# This is a function to add specific priviledges to process.
		# In our case we need the SeTakeOwnership privlidge which isn't granted to processes by default.
		
			###############################
			#Win32Constants
			###############################
		
			$Constants = @{
				SE_PRIVILEGE_ENABLED = 0x00000002;
				TOKEN_QUERY = 0x00000008;
				TOKEN_ADJUST_PRIVILEGES = 0x00000020;
			}

			$Win32Constants = New-Object PSObject -Property $Constants
			try
			{
				$Result = $true
				Switch ($true){
					($Result) {
						# Get current process handle
						[IntPtr] $hproc = [Win32API]::GetCurrentProcess()
						[IntPtr] $htok = [IntPtr]::Zero
						# Open token handle with TOKEN_ADJUST_PRIVILEGES bor TOKEN_QUERY
						$Result = [Win32API]::OpenProcessToken($hproc, ($Win32Constants.TOKEN_ADJUST_PRIVILEGES -bor $Win32Constants.TOKEN_QUERY), [ref] $htok)
					}
					($Result) {
						# Luid
						$LuidVal = $null
						# Get Luid for privelege
						$Result = [Win32API]::LookupPrivilegeValue($null, $Privilege, [ref] $LuidVal)
					}
					($Result) {
						# Creating the TokPriv1Luid structure
						[TokPriv1Luid]$TokPriv1Luid = New-Object TokPriv1Luid -Property @{
							Count = 1 # Amount of priveleges
							Luid = $LuidVal # Luid
							Attr = $Win32Constants.SE_PRIVILEGE_ENABLED 
						}
						# Enable privelege for the current process
						$Result = [Win32API]::AdjustTokenPrivileges($htok, $false, [ref] $TokPriv1Luid, 0, [IntPtr]::Zero, [IntPtr]::Zero)
					}
				}
				
				return $Result
			}
			finally
			{
				# Free the handle of current process
				if ($htok -ne [IntPtr]::Zero) {
					If(![Win32API]::CloseHandle($htok)) {
						Write-Warning "Closing process handle failed, Win32Error: $([System.Runtime.InteropServices.Marshal]::GetLastWin32Error())"
					}
				}
			}
		}
		# Check if process running as TrustedInstaller
		$IsTrustedInstaller = $false
		if (([System.Security.Principal.WindowsIdentity]::GetCurrent()).Owner -ne "S-1-5-18") {
			ForEach ($sid in [security.principal.windowsidentity]::getcurrent().groups){
				if ($sid.translate([System.Security.Principal.NTAccount]) -eq "NT SERVICE\TrustedInstaller") {
					$IsTrustedInstaller = $true
				}
			}
		} else {$IsTrustedInstaller = $true}
		# Run process as TrustedInstaller
		if (!$IsTrustedInstaller) {
			
			# Checking SeDebugPrivilege privelege for current user
			$Whoami = whoami /priv /fo csv |ConvertFrom-Csv
			$SeDebugPriv = $whoami -Match "SeDebugPrivilege"
			
			$Result = $false
			# $SeDebugPriv = $false
			# SeDebugPriv needs to be available
			if ($SeDebugPriv) {
				# Starting TrustedInstaller service
				$SourceService = "TrustedInstaller"
				# Checking StartMode
				$ServiceObject = Get-WmiObject -Class win32_service | where-object { $_.name -eq $SourceService}
				if (($ServiceObject.StartMode -ne 'Manual') -and ($ServiceObject.StartMode -ne 'Automatic')) {
					# Changing StartMode to Manual
					$null = Set-Service -Name $SourceService -StartupType Manual -ErrorAction SilentlyContinue
				}
				# Checking StartMode
				$ServiceObject = Get-WmiObject -Class win32_service | where-object { $_.name -eq $SourceService}
				if (($ServiceObject.StartMode -eq 'Manual') -or ($ServiceObject.StartMode -eq 'Automatic')) {
					# Checking State
					if($ServiceObject.State -ne 'Running') {
						# Start service
						$null = Start-Service $SourceService -ErrorAction SilentlyContinue
						[int]$i = 0
						do {
							# Checking Status
							$sourceServiceObject = Get-Service $SourceService
							$i++
							Start-Sleep -Milliseconds 50
						} while(($sourceServiceObject.Status -ne 'Running') -and ($i -lt 10))
					}
					# Checking State
					$ServiceObject = Get-WmiObject -Class win32_service | where-object { $_.name -eq $SourceService}
					if ($ServiceObject.State -eq 'Running') {
						# Checking Service ID
						if ($ServicePid = $ServiceObject.ProcessId) {
							Write-Verbose "Source Service PID $ServicePid"
							# Get full path of the script
							[string]$FilePath = $null
							# Checking file path
							if ($FilePath = $(Get-CurrentLocation)["FilePath"]) {
								# Restart the script with priveleges of TrustedInstaller
								$ApplicationName = "$env:systemroot\System32\cmd.exe"
								$CommandLine = " /c `"$FilePath`" $args"
								write-host "Starting: " "$ApplicationName $CommandLine"
								# Run script
								$Result = CreateProcessFromParent $ServicePid $ApplicationName $CommandLine
							}
						}
					}
				}
			}

			if ($Result){
				# Run as TrustedInstaller is succeed
				[Win32API]::ShowWindow([Win32API]::GetConsoleWindow(), 0) # Hide current console window
				exit
			} else {
				# Run as TrustedInstaller is failed
				[string[]]$Privileges = @('SeTakeOwnershipPrivilege', 'SeBackupPrivilege', 'SeRestorePrivilege')
				# The privilege to adjust. This set is taken from
				# http://msdn.microsoft.com/en-us/library/bb530716(VS.85).aspx
				ForEach ($Privilege in $Privileges){
					# Enable privilege
					do {} until (Elevate-Privileges($Privilege))
				}
				# Get local name of admin
				$AdminLocalName = (gwmi win32_useraccount | Where-Object {$_.SID -match '^S-1-5-21-[\d-]+-500$'}).Name
				# Title of console
				[System.Console]::Title = "$AdminLocalName : Additional Windows Update Tools"
			}
		} else {
			# The process already running as TrustedInstaller
			[System.Console]::Title = "TrustedInstaller : Additional Windows Update Tools"
		}
		# ।饭 訡 ॢ襭 ᨬ쭮  ࠧ ᮫   ࠧ襭
		if ((120 -gt [System.Console]::LargestWindowWidth) -or (40 -gt [System.Console]::LargestWindowHeight)) {
			$WindowWidth = [System.Console]::WindowWidth
			$WindowHeight = [System.Console]::WindowHeight
		} else {
			$WindowWidth = 120
			$WindowHeight = 40
		}

		# ਭ  ᮫
		[System.Console]::BufferWidth = $WindowWidth
		#   ᮫
		[System.Console]::BufferHeight = 9000
		# ਭ  ᮫
		[System.Console]::WindowWidth = $WindowWidth
		#   ᮫
		[System.Console]::WindowHeight = $WindowHeight
	}
	
	function Set-RegPermissions {
		param([Microsoft.Win32.RegistryKey]$regKey = $null, [string]$RestoreSDDL = $null)
		# 祭 ࠢ 㯠    ॥
		if ($regKey) {
			if ($RestoreSDDL) {
				# ⠭ 室 ஥ 㯠,  
				$acl = $regKey.GetAccessControl()
				$acl.SetSecurityDescriptorSddlForm($RestoreSDDL)
				$regKey.SetAccessControl($acl)
			} else {
				# ࠭  室 ஥ 㯠
				[string]$RestoreSDDL = $regKey.GetAccessControl().Sddl
				
				# 祭 ᨤ ⥪饣 짮⥫
				$user = New-Object System.Security.Principal.NTAccount("$env:UserName")
				$Owner = $user.Translate([System.Security.Principal.SecurityIdentifier])

				#  
				$acl = $regKey.GetAccessControl()
				$acl.SetOwner($Owner)
				$acl.SetGroup($Owner)
				$regKey.SetAccessControl($acl)

				# ࠢ  祭  ࠢ
				$rule = New-Object System.Security.AccessControl.RegistryAccessRule (
					$Owner,
					'FullControl',
					'ContainerInherit',
					'None',
					'Allow'
				)
				
				# 祭  ࠢ
				$acl = $regKey.GetAccessControl()
				$acl.SetAccessRule($rule)
				$regKey.SetAccessControl($acl)

				# 室 ன 㯠
				return $RestoreSDDL
			}
		}
	}
	
	function Set-FilePermissions {
		param([System.IO.FileInfo]$FileInfo = $null, [string]$RestoreSDDL = $null)
		# 祭 ࠢ 㯠  䠩
		if ($FileInfo) {
			if ($RestoreSDDL) {
				# ⠭ 室 ஥ 㯠,  
				$acl = $FileInfo.GetAccessControl()
				$acl.SetSecurityDescriptorSddlForm($RestoreSDDL)
				$FileInfo.SetAccessControl($acl)
			} else {
				# ࠭  室 ஥ 㯠
				[string]$RestoreSDDL = $FileInfo.GetAccessControl().Sddl
				
				# 祭 ᨤ ⥪饣 짮⥫
				$user = New-Object System.Security.Principal.NTAccount("$env:UserName")
				$Owner = $user.Translate([System.Security.Principal.SecurityIdentifier])

				#  
				$acl = $FileInfo.GetAccessControl()
				$acl.SetOwner($Owner)
				$acl.SetGroup($Owner)
				$FileInfo.SetAccessControl($acl)

				# ࠢ  祭  ࠢ
				$rule = New-Object Security.AccessControl.FileSystemAccessRule (
					$Owner,
					'FullControl',
					'Allow'
				)
				
				# 祭  ࠢ
				$acl = $FileInfo.GetAccessControl()
				$acl.SetAccessRule($rule)
				$FileInfo.SetAccessControl($acl)

				# 室 ன 㯠
				return $RestoreSDDL
			}
		}
	}
	
	function Set-FolderPermissions {
		param([System.IO.DirectoryInfo]$DirectoryInfo = $null, [string]$RestoreSDDL = $null)
		# 祭 ࠢ 㯠  䠩
		if ($DirectoryInfo) {
			if ($RestoreSDDL) {
				# ⠭ 室 ஥ 㯠,  
				$acl = $DirectoryInfo.GetAccessControl()
				$acl.SetSecurityDescriptorSddlForm($RestoreSDDL)
				$DirectoryInfo.SetAccessControl($acl)
			} else {
				# ࠭  室 ஥ 㯠
				[string]$RestoreSDDL = $DirectoryInfo.GetAccessControl().Sddl
				
				# 祭 ᨤ ⥪饣 짮⥫
				$user = New-Object System.Security.Principal.NTAccount("$env:UserName")
				$Owner = $user.Translate([System.Security.Principal.SecurityIdentifier])

				#  
				$acl = $DirectoryInfo.GetAccessControl()
				$acl.SetOwner($Owner)
				$acl.SetGroup($Owner)
				$DirectoryInfo.SetAccessControl($acl)

				# ࠢ  祭  ࠢ
				$rule = New-Object Security.AccessControl.FileSystemAccessRule (
					$Owner,
					'FullControl',
					('ContainerInherit', 'ObjectInherit'),
					'None',
					'Allow'
				)
				
				# 祭  ࠢ
				$acl = $DirectoryInfo.GetAccessControl()
				$acl.SetAccessRule($rule)
				$DirectoryInfo.SetAccessControl($acl)

				# 室 ன 㯠
				return $RestoreSDDL
			}
		}
	}

	function pause ($Message = "Press any key to continue . . . ") {
		# 㧠
		if ((Test-Path variable:psISE) -and $psISE) {
			$Shell = New-Object -ComObject "WScript.Shell"
			$Button = $Shell.Popup("Click OK to continue.", 0, "Script Paused", 0)
		}
		else {
			Write-Host $Message
			[void][System.Console]::ReadKey($true)
			Write-Host
		}
	}

	function Get-OsDism {
		# । 䠩 DISM.exe
		$dism = @("$env:SystemRoot\sysnative\dism.exe", "$env:SystemRoot\system32\dism.exe", "dism.exe") | where { Test-Path -Path $_ } | Select-Object -First 1
		if(!$dism) {
			try {throw [System.ArgumentException] "Unable to locate DISM.exe"} catch {}
		}
		return $dism
	}

	function Get-OsArch {
		# । ⥪ ⥬
		[string]$arch = "amd64"
		if ($env:PROCESSOR_ARCHITECTURE -eq "x86") {
			if (!$env:PROCESSOR_ARCHITEW6432) {$arch = "x86"}
		}
		
		return $arch
	}

	function Get-OsVers {
		# । ᨨ ⥬
		[string[]]$VerArray = ((gwmi -class Win32_OperatingSystem).Version).split(".")
		$osver = $VerArray[0] + "." + $VerArray[1]
		
		return $osver
	}
	
	function Get-OsLangs
	{
		# 祭 ᯨ᪠ 몮 ⮢
		[string[]]$Languages = $null
		# ⨥ 
		$RegPackagesKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages")
		#   ᯨ ⮢
		ForEach ($PackageName in $RegPackagesKey.GetSubKeyNames()) {
			# 宦  몮  
			if ($PackageName -match "^Microsoft-Windows-Client-LanguagePack-Package~") {
				# 宦 䨪 몮  
				$Language = ($PackageName.split("~")[3])
				#   ᯨ᮪
				$Languages += $Language
			}
		}
		# ⨥ 
		$RegPackagesKey.Close()
		
		return $Languages
	}

	function Get-CurrentLocation ()
	{
		# 祭 ⥪饩 ४ਨ ਯ
		[string]$Location = $null
		# 祭 ᯨ᪠ 饭 ᮢ
		$ProcessArray = gwmi win32_process
		# 祭 ꥪ ⥪饣 
		$current = ($ProcessArray | Where-Object {$_.processid -eq $pid})
		# 祭 ꥪ த⥫᪮ 
		# $parent = ($ProcessArray | Where-Object {$_.processid -eq $current.parentprocessid})
		# 祭 ⥪饩 ४ਨ
		$FilePath = $null
		switch ($true) {
			([string]::IsNullOrEmpty($FilePath)) { 
				# ࠡ⪠ 맮  㬥⠬
				$FilePath = (($current.commandline).Split('"') | Where-Object {(($_ -match "\.cmd") -or ($_ -match "\.ps1"))})
			}
			([string]::IsNullOrEmpty($FilePath)) {
				# ࠡ⪠ 맮  㬥⮢
				$FilePath = $script:MyInvocation.MyCommand.Path
			}
			(![string]::IsNullOrEmpty($FilePath)) {
				# 塞 ᨬ "\"   
				$FilePath = (($FilePath.Split('\') | Where-Object {$_ -ne ""}) -join("\"))
				#  ४
				$Location = (Split-Path $FilePath) + "\"
			}
		}
				
		return @{"Location" = $Location; "FilePath" = $FilePath}
	}

	function Out-Log ([System.Text.StringBuilder]$sb = $null)
	{
		# 뢮   
		try {
			# ᪫  ࠬ -WhatIf
			$tmpWhatIfPreference = $WhatIfPreference
			$WhatIfPreference = $false
			if (($LogFile) -and ($sb)) {
				#    
				$LogsPath = [System.IO.Path]::GetDirectoryName("$LogFile")
				if (![System.IO.File]::Exists("$LogsPath")) {[void][System.IO.Directory]::CreateDirectory("$LogsPath")}
				[System.IO.File]::AppendAllText("$LogFile", $sb, [System.Text.Encoding]::UTF8)
			}
		} finally {
			$WhatIfPreference = $tmpWhatIfPreference
		}
	}

	function Get-Compare ([string] $val_cur, [string] $val_pre)
	{
		#  襣 祭
		if (!$val_pre) {
		# ᢮ ࢮ 祭
			$val_new=$val_cur
		}	
		else {
			$val_new=$val_pre
			#  ப  ஬ ᨨ
			$items_cur = $val_cur.split(".")
			$items_pre = $val_pre.split(".")
			# ࠢ ᨩ
			for ($n = 0; $n -lt $items_cur.count; $n++) {
				# 祭 ஢ 祭
				$item_pre = [int]$items_pre[$n]
				$item_cur = [int]$items_cur[$n]
				# ᫨ 祭  ࠢ
				if ($item_pre -ne $item_cur) {
					# ᫨ ।饥 祭  
					if ($item_pre -le $item_cur) {
						$val_new=$val_cur
						Break
					}
					# ᫨ ।饥 祭  
					else {Break}
				}
			}
		}
		return $val_new	
	}

	function Get-Version ([string[]]$Components, [string]$ext_ver = "0.0")
	{
		# । 㠫쭮 ᨨ 
		[string]$ver = $null
		#  ᨩ  
		[System.Collections.Hashtable]$ArrayVers = @{}
		#   祭 
		ForEach ($Component in $Components)
		{
			[string]$key = $null
			[string[]]$values = $null
			# ⨥ 
			$RegBaseKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$Component")
			#   
			if ($RegBaseKey) {
				#  祭  㬮砭   ࠧ
				$values = $RegBaseKey.GetValue("")
				# ᫨ 祭  㬮砭  ࠧ  ᢮
				if (!$values) {
					# 祭 ᯨ᪠  ࠧ
					$values = $RegBaseKey.GetSubKeyNames()
				}	
				if ($values) {
					#   ࠬ ࠧ
					ForEach ($v in $values) {
						[string]$version = $null
						$key = "$Component" + "\" + "$v"
						# ⨥ 
						$RegSubKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$key")
						#  祭  㬮砭   ᨨ 
						$version = $RegSubKey.GetValue("")
						# ᫨ 祭  㬮砭  ᨨ   ᢮
						if (!$version) {
							#  ᨨ 
							[string[]]$versions = @()
							# 祭 ᯨ᪠  ᨩ
							[string[]]$versions = ForEach ($val in $RegSubKey.GetValueNames()){$val.Split('',[System.StringSplitOptions]::RemoveEmptyEntries)}
							# 롮 ᫥ 祭
							$version = $versions[$versions.Count-1]
						}
						# ᫨  ࠧ 㦥   ⠡
						if ($ArrayVers.ContainsKey("$v")) {
							$val_ver = $ArrayVers["$v"]
							# 롮 襣 祭
							$version = Get-Compare $version $val_ver
						}
						$ArrayVers["$v"] = $version
						# ⨥ 
						$RegSubKey.Close()
					}
				}
				# ⨥ 
				$RegBaseKey.Close()
			}
		}
		if($ArrayVers.Count){
			# ࠧ  । ᨥ
			if ($ext_ver -ne "0.0") {
				$ver = $ArrayVers["$ext_ver"]
			} else {
				# ࠧ  襩 ᨥ
				ForEach ($v in $ArrayVers.Keys) {
					$value = Get-Compare $v $value
				}
				$ver = $ArrayVers["$value"]
			}
		}
		#  ᨨ
		return $ver
	}

	function Get-StateCode ([string]$Status)
	{
		#   
		switch ("$Status") {
			"Uninstall Pending"		{ $StatusCode = 5 }
			"Not Present"			{ $StatusCode = 32 }
			"Staged"				{ $StatusCode = 64 }
			"Superseded"			{ $StatusCode = 80 }
			"Install Pending"		{ $StatusCode = 96 }
			"Installed"				{ $StatusCode = 112 }
			Default					{ $StatusCode = 0 }
		}
		return $StatusCode
	}

	function Get-ComponentPackages ([string]$Title, [System.Collections.Hashtable]$ComponentPackages, [System.Text.StringBuilder]$Log = $null)
	{
		# ᮪ ଠ ⮢
		[System.Collections.SortedList]$PermanentPackages = @{}
		# ᮪  ࠧ஢묨 ⠬
		[System.Collections.SortedList]$UnlockedPackages = @{}
		# ᮪ ॣ ⮢
		[System.Collections.SortedList]$RegularPackages = @{}
		# ᮪ ⮢  
		if ($ComponentPackages.Count) {
			#  
			$RegularPackages = $ComponentPackages.Clone()
			# । ଠ ⮢
			$PermanentPackages = Get-PermanentPackages $ComponentPackages
			# ଠ 
			if ($PermanentPackages.Count) {
				# 롮  ⮢
				ForEach ($p in $PermanentPackages.Keys) {
					$RegularPackages.Remove("$p")
				}
			}
		}
		# 뢮 १⮢  ⮢
		Out-PackagesResults $Title $RegularPackages.Keys $PermanentPackages.Keys $Log
		# 頥 祭
		$ComponentPackages = $RegularPackages.Clone()
		# ࠡ⪠ ଠ ⮢
		if ($PermanentPackages.Count) {
			# ।  ࠧ஢ ⮢
			if (!(Out-Menu_Break 'Unlock permanent packages' 'Skip permanent packages')) {
				# ஢ ⮢
				$UnlockedPackages = Unlock-PermanentPackages $PermanentPackages $Log
			}
			# । ⮢,   뫨 ࠧ஢
			if ($UnlockedPackages.Count) {
				# । ଠ ⮢
				$PermanentPackages = Get-PermanentPackages $UnlockedPackages
				# ଠ 
				if ($PermanentPackages.Count) {
					# ⡮ ࠧ஢ ⮢
					ForEach ($p in $PermanentPackages.Keys) {
						$UnlockedPackages.Remove("$p")
					}
					#  ଠ⭮ ⪨ ⠬,   뫨 ࠧ஢
					$Title = 'Failed'
					$Target = ' permanence="removable"'
					$Replacement = ' permanence="permanent"'
					$PermanentPackages = Unlock-PermanentPackages $PermanentPackages $Log $Title $Target $Replacement
				}
			}
			# ஢ 
			if ($UnlockedPackages.Count) {
				#  ࠧ஢ ⮢
				ForEach ($p in $UnlockedPackages.Keys) {
					$ComponentPackages["$p"] = $UnlockedPackages["$p"]
				}
			}
		}
		
		return $ComponentPackages
	}

	function Get-PermanentPackages ([System.Collections.SortedList]$ComponentPackages, [string]$Target = ' permanence="permanent"')
	{
		# ᮪ ଠ 
		[System.Collections.SortedList]$PermanentPackages = @{}
		# । ଠ ⮢
		if ($ComponentPackages.Count) {
			#   ᭮ ⠬  
			ForEach ($PrimaryPackage in $ComponentPackages.Keys) {
				# ஢ઠ 
				$IsPermanent = $false
				#   ⠬ 栬
				ForEach ($Owner in $ComponentPackages["$PrimaryPackage"]["Owners"].Keys) {
					#   ⠬  
					ForEach ($PackageName in $ComponentPackages["$PrimaryPackage"]["Owners"]["$Owner"].Keys) {
						# ⪨ 
						if (!$IsPermanent) {
							#  
							$file = "$PackagesFolder\$PackageName.mum"
							# ஢ઠ 䠩
							if ([System.IO.File]::Exists("$file")) {
								#  ⪨
								[System.IO.StreamReader]$reader = "$file"
								$IsExit = $false
								While (!$reader.EndOfStream) {
									$line = $reader.ReadLine()
									if ($line -match "$Target") {
										$IsExit = $true
										#  ଠ
										$IsPermanent = $true
									}
									if ($IsExit) {Break}
								}
								$reader.Close()
							}
						}
					}
				}
				if ($IsPermanent) {
					# ଠ⭮ 
					$PermanentPackages["$PrimaryPackage"] = $ComponentPackages["$PrimaryPackage"]
				}
			}
		}
		
		return $PermanentPackages
	}

	function Unlock-PermanentPackages ([System.Collections.SortedList]$PermanentPackages, [System.Text.StringBuilder]$Log = $null, [string]$Title = 'Unlocking', [string]$Target = ' permanence="permanent"', [string]$Replacement = ' permanence="removable"')
	{
		# C᮪ ࠧ஢ ⮢
		[System.Collections.SortedList]$UnlockedPackages = @{}
		# C᮪ ७ ⮢ ( ᢮⢮ "Trusted" =dword:00000001)
		[System.Collections.SortedList]$TrustedPackages = @{}
		# , 祭 ᢮⢠    㤠
		[System.Collections.SortedList]$UnchangedPackages = @{}
		# 
		$Color = 'Green'
		if ("$Title" -ne 'Unlocking') {$Color = 'Red'}
		# ࠡ⪠ ଠ ⮢
		if ($PermanentPackages.Count) {
			#   ᭮ ⠬  
			ForEach ($PrimaryPackage in $PermanentPackages.Keys) {
				# 뢮  ࠧ㥬 
				Write-Host "$Title : " -ForegroundColor $Color -NoNewline
				Write-Host "$PrimaryPackage"
				# 
				if ($Log) {[void]$Log.AppendLine("$Title : $PrimaryPackage")}
				#   ⠬ 栬
				ForEach ($Owner in $PermanentPackages["$PrimaryPackage"]["Owners"].Keys) {
					#   ⠬  
					ForEach ($PackageName in $PermanentPackages["$PrimaryPackage"]["Owners"]["$Owner"].Keys) {
						#    ᯨ᮪ ७
						$TrustedPackages["$PackageName"] = "$Owner"
					}
				}
				# ᢮ ⠬ ᢮⢠ "Trusted"  ଠ쭮 㤠
				$UnchangedPackages = Set-PackageProperty $TrustedPackages 'Trusted' 1
				# ⢮ "Trusted"    ⮢
				if (!$UnchangedPackages.Count) {
					#   ⠬  
					ForEach ($PackageName in $TrustedPackages.Keys) {
						#   䠩 饩 ⪨
						$file = "$PackagesFolder\$PackageName.mum"
						# ஢ઠ 䠩
						if ([System.IO.File]::Exists("$file")) {
							# 室 ࠢ 㯠
							[string]$FileSDDL = $null
							if ($PSCmdlet.ShouldProcess("$file", "$Title Permanent Package")) {
								# 祭 㯠  䠩
								if(!$IsTrustedInstaller) {
									[System.IO.FileInfo]$FileInfo = New-Object System.IO.FileInfo ("$file")
									$FileSDDL = Set-FilePermissions $FileInfo
								}
								#   䠩 饩 ⪨
								[System.IO.File]::WriteAllText("$file", [System.IO.File]::ReadAllText("$file").Replace("$Target", "$Replacement"))
								# 饭 室   ⬥ ࠢ 㯠
								if ($FileSDDL) {Set-FilePermissions $FileInfo $FileSDDL}
							}
						}
					}
					#  ࠧ஢ ⮢  ᯨ᮪
					$UnlockedPackages["$PrimaryPackage"] = $PermanentPackages["$PrimaryPackage"]
				}
			}
		}

		return $UnlockedPackages
	}
	
	function Set-PackageProperty ()
	{
		# ᢮ ᢮ ⮢ ࠭ 祭
		Param (
		  [Parameter(Mandatory=$True,Position=0)]
		  [System.Collections.SortedList]$CurrentPackages,
		  
		  [Parameter(Mandatory=$True,Position=1)]
		  [string]$Property,
		  
		  [Parameter(Mandatory=$True,Position=2)]
		  $PropertyValue,
		  
		  [Parameter(Mandatory=$False,Position=3)]
   		  [ValidateSet("String", "ExpandString", "MultiString", "DWord", "QWord", "Binary")]
		  [string]$PropertyType = 'DWord'
		)

		# ,  ,  㤠 
		[System.Collections.SortedList]$UnchangedPackages = @{}
		
		#  Owners
		$IsOwner = $false
		#   ஢ 
		if ("$Property" -eq 'Permanence') {$IsOwner = $true}
		
		if ($CurrentPackages.Count) {
			#   ⠬
			ForEach ($Package in $CurrentPackages.Keys) {
				# 롮  Owners     ⢥ ᢮⢠
				if ($IsOwner) {$Property = $Package; $Package = "$Package" + '\Owners'}
				# ⨥  
				[string]$RegKeySDDL = $null
				$RegKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$Package",'ReadSubTree','QueryValues')
				# ᫨  
				if ($RegKey) {
					# 祭 ࠭ ᢮⢠
					$CurrentValue = $RegKey.GetValue("$Property")
					# 祭 ᢮⢠  ᮢ  ॡ㥬
					if ("$CurrentValue" -ne "$PropertyValue") {
						#    祭 ࠢ 㯠
						if(!$IsTrustedInstaller) {
							$RegKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$Package",'ReadWriteSubTree','TakeOwnership')
							$RegKeySDDL = Set-RegPermissions $RegKey
						}
						#  祭
						$RegKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$Package",'ReadWriteSubTree','FullControl')
						$RegKey.SetValue("$Property","$PropertyValue","$PropertyType")
						# 饭 室   ⬥ ࠢ 㯠
						if ($RegKeySDDL) {Set-RegPermissions $RegKey $RegKeySDDL}
					}
					# ஢ઠ 祭 ࠭ ᢮⢠
					if ($RegKey.GetValue("$Property") -ne "$PropertyValue") {
						# , 祭  ,  㤠 
						$UnchangedPackages["$Package"] = $CurrentPackages["$Package"]
					}

					# 뢠  
					$RegKey.Close()
				} else {
					# ,  ,  㤠 
					$UnchangedPackages["$Package"] = $CurrentPackages["$Package"]
				}
			}
		}
		return $UnchangedPackages
	}

	function Start-DeletePackages ([System.Collections.SortedList]$DeletePackages, $Status = "Installed")
	{
		# ஢塞 稥 ⮢  㤠  ⠭
		if (![Microsoft.Win32.Registry]::$RootHive.OpenSubKey("SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\PackagesPending")) {
			#   ⮢  "Superseded"
			[string[]]$ProcessedPackages = $null
			# ,   㤠 ࠧ஢
			[System.Collections.SortedList]$BlockedPackages = @{}
			#   ଠ ᮬ (  )
			[System.Collections.SortedList]$UnchangedPackages = @{}
			#   ᮬ "Superseded"
			[System.Collections.SortedList]$SucceededPackages = @{}
			# ஢
			$SucceededPackages = $DeletePackages.Clone()
			
			# ᢮ ⠬ ࠭ 
			if ($DeletePackages.Count) {
				# ⨥ ஢ ⮢
				$BlockedPackages = Set-PackageProperty $DeletePackages 'Permanence' 131184
			}
			if ($BlockedPackages.Count) {
				#   騬 ᫮ ⮢  ⥬ ⠬,   뫨 ࠧ஢
				ForEach ($p in $BlockedPackages.Keys){$SucceededPackages.Remove("$p")}
			}
			if ($SucceededPackages.Count) {
				#   
				$StatusCode = Get-StateCode 'Superseded'
				# ᢮ ⠬ ࠭ 
				$UnchangedPackages = Set-PackageProperty $SucceededPackages 'CurrentState' $StatusCode
			}
			if ($UnchangedPackages.Count) {
				#   騬 ᫮ ⮢  ⥬ ⠬,   㤠  
				ForEach ($p in $UnchangedPackages.Keys){$SucceededPackages.Remove("$p")}
				Write-Host
				Write-Host "Packages that wasn't unlocked" -ForegroundColor Red
				$BlockedPackages.Keys | Out-Host
				Write-Host
				Write-Host "Packages with Unchanged Status" -ForegroundColor Red
				$UnchangedPackages.Keys | Out-Host
				# ।  祭 ॣ ⮢
				if (!(Out-Menu_Break 'Append regular packages' 'Skip regular packages')) {$SucceededPackages = $DeletePackages}
			}
			#  ⮢
			if ($SucceededPackages.Count) {
				#  ⮢  ᯨ᮪
				ForEach ($p in $SucceededPackages.Keys) {
					$ProcessedPackages += "/PackageName:$p"
				}
				# ꥤ  ⮢   ப
				$PackagesValue = [system.String]::Join(" ", $ProcessedPackages)
				# । 䠩 DISM.exe
				$dism = Get-OsDism
				#  ⮢
				if ($dism) {
					$DismArguments = @("/online", "/Remove-Package $PackagesValue", "/norestart")
					if ($PSCmdlet.ShouldProcess("$($SucceededPackages.Count) Packages", "Uninstalling")) {
						$ExitCode = (Start-Process -FilePath "$dism" -ArgumentList $DismArguments -Wait -NoNewWindow -PassThru).ExitCode
					}
				}
				# ⠭   㤠 ⮢
				if ($ExitCode -ne 0) {
					#   
					$StatusCode = Get-StateCode "$Status"
					# ᢮ ⠬ ࠭ 
					$UnchangedPackages = Set-PackageProperty $SucceededPackages 'CurrentState' $StatusCode
				}
				#   室 १㧪
				Write-Host
				Write-Host "Please restart the system to complete uninstall operation" -ForegroundColor Yellow
				Write-Host
			} else {
				# ᫨ 祣 㤠
				Write-Host
				Write-Host "There are no packages to remove." -ForegroundColor Yellow
				Write-Host
			}
		} else {write-host "Pending operations exist, please restart the system" -ForegroundColor Red}
	}

	function Out-UpdatesResults ([string]$Title, [string[]]$UpdatesList, [string[]]$ActualUpdates, [System.Collections.Hashtable]$SupersededUpdates, [System.Collections.Hashtable]$PackageOwners, [System.Text.StringBuilder]$Log = $null)
	{
		# 뢮 १⮢
		if ($UpdatesList){
			Write-Host
			Write-Host "Component: " -ForegroundColor Green -NoNewline
			Write-Host $Title -ForegroundColor Yellow
			Write-Host
			# 뢮 㠫쭮 
			if ($ActualUpdates){
				# ᮪ 㠫    ᭮ ⮢
				[System.Collections.SortedList]$RelevantUpdates = @{}
				# ।  㠫쭮 
				ForEach ($u in $ActualUpdates) {
					#  
					$UpdateName = $PackageOwners["Updates"]["$u"]["UpdateName"]
					$RelevantUpdates["$UpdateName"] = $u
				}
				# 㠫쭮 
				$ActualUpdate = $RelevantUpdates.GetKey($RelevantUpdates.count - 1)
				Write-Host "Actual update is : " -ForegroundColor Cyan -NoNewline
				Write-Host $ActualUpdate
				Write-Host
			}
			# 뢮 㤠塞 
			if ($UpdatesList){
				if ($SupersededUpdates){
					Write-Host "List of updates to proceed" -NoNewline -ForegroundColor Cyan
					Write-Host " (Replaced) " -NoNewline -ForegroundColor DarkGray
					Write-Host ':' -ForegroundColor Cyan
					Write-Host
					ForEach ($u in $UpdatesList) {
						#  
						Write-Host $SupersededUpdates["$u"]["UpdateName"] -NoNewline
						# ᮪     
						Write-Host '	by :' @(ForEach ($a in $SupersededUpdates["$u"]["ActualUpdates"].Keys) {if ($PackageOwners["Updates"]["$a"]) {$PackageOwners["Updates"]["$a"]["UpdateName"]} }) -ForegroundColor DarkGray
					}
					Write-Host
				} else {
					Write-Host "List of updates to proceed :" -ForegroundColor Cyan
					Write-Host
					$UpdatesList | Out-Host
					Write-Host
				}
			}
			#   
			$SolidWidth = ($Host.UI.RawUI.WindowSize).Width - 1
			$SolidLine = "-" * $SolidWidth
			Write-Host $SolidLine
			
			if ($Log) {
				# 뢮 १⮢  
				[void]$Log.AppendLine().AppendLine("Component: $Title").AppendLine()
				# 뢮 㠫쭮 
				if ($ActualUpdates){
					# 
					[void]$Log.AppendLine("Actual update is : $ActualUpdate").AppendLine()
				}
				# 뢮 㤠塞 
				if ($UpdatesList){
					if ($SupersededUpdates){
						# 
						[void]$Log.AppendLine("List of updates to proceed  (Replaced) :").AppendLine()
						ForEach ($u in $UpdatesList) {
							#  
							[void]$Log.Append($PackageOwners["Updates"]["$u"]["UpdateName"]).Append('	by : ')
							# ᮪     
							ForEach ($a in $SupersededUpdates["$u"]["ActualUpdates"].Keys) {
								if ($PackageOwners["Updates"]["$a"]) {
									[void]$Log.Append($PackageOwners["Updates"]["$a"]["UpdateName"]).Append(' ')
								}
							}
							[void]$Log.AppendLine()
						}
					} else {
						# 
						[void]$Log.AppendLine("List of updates to proceed :").AppendLine()
						ForEach ($u in $UpdatesList) {[void]$Log.AppendLine($u)}
						[void]$Log.AppendLine()
					}
				}
				#   
				[void]$Log.AppendLine().AppendLine($SolidLine)
			}
		}
	}

	function Out-PackagesResults ([string]$Title, [string[]]$RegularPackages, [string[]]$PermanentPackages, [System.Text.StringBuilder]$Log = $null)
	{
		# 뢮 १⮢  ⮢
		if ($RegularPackages -or $PermanentPackages){
			Write-Host
			Write-Host "Component: " -ForegroundColor Green -NoNewline
			Write-Host $Title -ForegroundColor Yellow
			Write-Host
			# 뢮 㤠塞 ⮢
			if ($RegularPackages){
			Write-Host "List of packages to proceed :" -ForegroundColor Cyan
			Write-Host
			$RegularPackages | Out-Host
			Write-Host
			}
			# 뢮 ଠ ⮢
			if ($PermanentPackages) {
				Write-Host "List of permanent packages :" -ForegroundColor Red
				Write-Host
				$PermanentPackages | Out-Host
				Write-Host
			}
			#   
			$SolidWidth = ($Host.UI.RawUI.WindowSize).Width - 1
			$SolidLine = "-" * $SolidWidth
			Write-Host $SolidLine

			if ($Log) {
				# 뢮 १⮢  
				if ($RegularPackages -or $PermanentPackages){
					[void]$Log.AppendLine().AppendLine("Component: $Title").AppendLine()
					# 뢮 㤠塞 ⮢
					if ($RegularPackages){
						[void]$Log.AppendLine("List of packages to proceed :").AppendLine()
						ForEach ($p in $RegularPackages) {
							[void]$Log.AppendLine("$p")
						}
					}
					# 뢮 ଠ ⮢
					if ($PermanentPackages) {
						[void]$Log.AppendLine("List of permanent packages :").AppendLine()
						ForEach ($p in $PermanentPackages) {
							[void]$Log.AppendLine("$p")
						}
					}
					#   
					[void]$Log.AppendLine().AppendLine($SolidLine)
				}
			}
		}
	}
	
	function Get-PackagesList ([string[]]$UpdateList, [string]$filter = '^#', [bool]$IsVers = $false, [bool]$IsOut = $false, [bool]$IsStubs = $false, [bool]$IsState = $false)
	{
		# 祭 ᯨ᪠ ⮢, ⠭  ⥬
		if ($IsOut) {Write-Host 'Get updates list. Please wait...'}
		# ᮪  ⮢   ⠡
		$NormalRegularPackages = @();	
		# ᮪ ﭭ  ⮢   ⠡
		$NormalPermanentPackages = @();
		# ᮪  ⮢   ⠡
		$HiddenRegularPackages = @();
		# ᮪ ﭭ  ⮢   ⠡
		$HiddenPermanentPackages = @();
		
		# 
		[System.Collections.Hashtable]$PackageOwners = @{}
		# 騩 ᯨ᮪  ⮢  ன
		[System.Collections.SortedList]$PackageOwners["Packages"] = @{}
		# ᮪ ᭮ ⮢  ன ୨ ⮢
		[System.Collections.SortedList]$PackageOwners["Primary"] = @{}
		# 롮ઠ ᭮ ⮢  ன ୨ ⮢
		[System.Collections.SortedList]$PackageOwners["Updates"] = @{}
		# ᮪ ⮢,   栬  㣨 ⮢
		[System.Collections.SortedList]$PackageOwners["Owners"] = @{}
		# ᮪ ⮢,    栬  㣨 ⮢
		[System.Collections.SortedList]$PackageOwners["Childs"] = @{}
		# 쭠 롮ઠ ᭮ ⮢     ᯨ᪠
		[System.Collections.SortedList]$PackageOwners["Custom"] = @{}
		# ᪫砥 ᨨ ⮢  짮⥫᪮ ᯨ᪠ 
		[System.Collections.SortedList]$PackageOwners["Names"] = @{}
				
		# ⨥  "HKLM\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\Packages"
		$RegPackagesKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages")
		#   ᯨ ⮢
		ForEach ($PackageName in $RegPackagesKey.GetSubKeyNames()) {
			# ᪫祭 ⨢ ⮢
			if ($PackageName -notmatch "$filter") {
				# ⨥  
				if ($RegOwnersKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$PackageName\Owners")) {
					# ନ஢ 樨 ⮢  
					ForEach ($PackageOwner in $RegOwnersKey.GetValueNames()) {
						#   ⮢,   栬  㣨 ⮢
						if (!$PackageOwners["Owners"]["$PackageOwner"]){[System.Collections.SortedList]$PackageOwners["Owners"]["$PackageOwner"] = @{}}
						$PackageOwners["Owners"]["$PackageOwner"]["$PackageName"] = "$PackageOwner"
						#  ⥪饣  ᮢ   
						if ("$PackageOwner" -eq "$PackageName") {
							#    ᯨ᮪ ᭮
							[System.Collections.SortedList]$PackageOwners["Primary"]["$PackageOwner"] = @{}
							# ᮪ ⮢,   栬  㣨 ⮢
							[System.Collections.SortedList]$PackageOwners["Primary"]["$PackageOwner"]["Owners"] = @{}
							# ᮪ ⮢,    栬  㣨 ⮢
							[System.Collections.SortedList]$PackageOwners["Primary"]["$PackageOwner"]["Packages"] = @{}
						}
					}
					#    騩 ᯨ᮪
					$PackageOwners["Packages"]["$PackageName"] = "$PackageName"
				}
			}
		}
		#   ᯨ ᭮ ⮢
		ForEach ($PackageOwner in $PackageOwners["Primary"].Keys) {
			# ᮪ ⮢ 楢  ᭮ 
			[System.Collections.SortedList]$TargetOwners = @{}
			#  ᭮   ᯨ᮪
			$TargetOwners["$PackageOwner"] = $false
			# ᭮ 
			$PrimaryName = $PackageOwner
			#   ᯨ 楢
			While ($PackageOwner -ne $null) {
				#   ୨ ⠬ 
				ForEach ($PackageName in $PackageOwners["Owners"]["$PackageOwner"].Keys) {
					#   ⪮  ⥫쭮 
					if ("$PackageOwner" -eq "$PrimaryName") {
						#    ⪮
						if ($PackageName -match "(.*)(?:-Wrapper)(.*)") {$PackageName = $matches[1] + $matches[2]}
					}
					#     饬 ᯨ᪥ ⮢
					if ($PackageOwners["Packages"]["$PackageName"]) {
						#    ᯨ᪥ 楢
						if ($PackageOwners["Owners"]["$PackageName"]) {
							#   ᯨ᮪ ⮢ 楢  ᭮ 
							$TargetOwners["$PackageName"] = $false
						}
					}
				}
				#  砥  ࠡ⠭
				$TargetOwners["$PackageOwner"] = $true
				#  楫    ᭮ 
				$PackageOwners["Primary"]["$PrimaryName"]["Owners"]["$PackageOwner"] = $PackageOwners["Owners"]["$PackageOwner"]
				# 롮 ᫥饣  ( ४ᨨ)
				$PackageOwner = $null
				ForEach ($Package in $TargetOwners.Keys) {
					#   ࠡ⠭
					if (!$TargetOwners["$Package"]) {
						$PackageOwner = "$Package"
						Break
					}
				}
			}
		}
		
		#   ᯨ ᭮ ⮢
		ForEach ($PrimaryName in $PackageOwners["Primary"].Keys) {
			#   ᯨ᮪
			$IsAppend = $false
			# ⥫쭮 
			$IsRollup = $false
			#  
			$UpdateName = "$null"
			#  
			$UpdateVers = "$null"
			#   ᮮ⢥ 㦭 
			if ($PrimaryName -match "(?:Package.*_for_)([^~_]*)(?:.*~~)([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*$)") {
				# 祭  
				$UpdateName = $matches[1]
				# 祭 ᨨ 
				$UpdateVers = $matches[2]
				# ⥫쭮 
				if ($UpdateName -match "Rollup") {
					# 㫥  
					$UpdateName = "$null"
					#  ࢮ  ᮮ⢥ 㦭 
					if ($PackageOwners["Primary"]["$PrimaryName"]["Owners"]["$PrimaryName"].GetKey(0) -match "KB[0-9]*") {
						#  ⥫쭮 
						$UpdateName = $matches[0]
					} else {
						#  ⥫쭮   ।
						$IsRollup = $true
					}
				}
				#   ⥫   Windows 10 1903   (६)
				# if($UpdateName){$IsAppend = $true}
				$IsAppend = $true
			}
			#  
			if (($IsRollup) -or ($IsOut) -or ($IsStubs)) {
				# 宦  
				$IsExit = $false
				# ஢ઠ 䠩
				if ([System.IO.File]::Exists("$PackagesFolder\$PrimaryName.mum")) {
					# ⨥ ⮪  ⥭ 䠩
					[System.IO.StreamReader]$reader = "$PackagesFolder\$PrimaryName.mum"
					# 筮 ⥭
					While (!$reader.EndOfStream) {
						$line = $reader.ReadLine()
						# 祭  
						if (!$UpdateName) {
							if ($line -match 'package identifier="KB[0-9]*"') {
								$UpdateName = $matches[0].split("=")[1].replace('"', '')
							}
						}
						# 祭 ⨯ 
						if ($line -match 'releaseType="(?i)[a-z ]*"') {
							$ReleaseType = $matches[0].split("=")[1].replace('"', '')
							if ($UpdateName) {$IsExit = $true}
						}
						if ($IsExit) {Break}
					}
					# ⨥ ⮪  ⥭
					$reader.Close()
				}
			}
			if (($IsOut) -or ($IsState)) {
				# 祭 ᢮ 
				$KeyValue = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$PrimaryName")
				
				#  
				$CurrentState = $KeyValue.GetValue('CurrentState')
				switch ("$CurrentState") {
					"5"		{ $PackageStatus = 'Uninstall Pending' }
					"32"	{ $PackageStatus = 'Not Present' }
					"64"	{ $PackageStatus = 'Staged' }
					"80"	{ $PackageStatus = 'Superseded' }
					"96"	{ $PackageStatus = 'Install Pending' }
					"112"	{ $PackageStatus = 'Installed' }
					Default { $PackageStatus = 'Unknown' }
				}
			}
			if ($IsOut) {			
				#  
				$Visibility = $KeyValue.GetValue('Visibility')
				# 塞  ᯨ᮪
				if ("$Visibility" -eq "1") {$Visibility = 'True'} else {$Visibility = 'False'}
				
				# ६ ⠭
				if (($InstallTimeHigh = $KeyValue.GetValue('InstallTimeHigh')) -and ($InstallTimeLow = $KeyValue.GetValue('InstallTimeLow'))){
					# ᫨ 祭  ॥ ॢ蠥 ([int32]::MaxValue = 2147483647),  頥 ⥫쭮 ᫮,
					# ஥   ([uint32]::MaxValue = 4294967295),  dword  [uint32] (⭮ )
					$InstallTimeLow = [System.BitConverter]::ToUInt32([System.BitConverter]::GetBytes($InstallTimeLow), 0)
					# 18-digit LDAP/FILETIME timestamp (1000000000 ns / 100); 1 sec = 1000000000 nanoseconds
					[long]$Timestamp = ([Math]::Pow(2,32) * $InstallTimeHigh + $InstallTimeLow)
					# The LDAP/FILETIME timestamp is the number of 100-nanoseconds intervals since Jan 1, 1601 UTC.
					try {$InstallTime = [System.DateTime]::Parse("1/1/1601").AddSeconds($TimeStamp/10000000).ToString("g")}
					# try {$InstallTime = Get-Date ((Get-Date 1/1/1601).AddSeconds($TimeStamp/10000000)) -Format g}
					catch {$InstallTime = ''; $error.RemoveAt(0)}
				}
				
				# 祭 ᢮  
				$RegOwnersKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackages\$PrimaryName\Owners")
				# ﭭ 
				$Permanence = 'False'
				if (($RegOwnersKey.GetValue("$PrimaryName")) -eq 131200) {$Permanence = 'True'}
								
				# ⢠ 
				[System.Collections.Hashtable]$Package = @{"|" = '|'; "Package Identity" = "$PrimaryName";	"  Name   " = "$UpdateName"; "  State  " = "$PackageStatus"; "Release Type" = "$ReleaseType"; "Install Time" = "$InstallTime"}
				#   ⠡ ⮢
				if ("$Visibility" -eq 'True') {
					#  
					if ("$Permanence" -eq 'False') {
						#   
						$NormalRegularPackages += $Package
					} else {
						# ﭭ 
						$Package["Permanence"] = "$Permanence"
						#  ﭭ  
						$NormalPermanentPackages += $Package
					}
				} else {
					#   
					$Package["Visibility"] = "$Visibility"
					#  
					if ("$Permanence" -eq 'False') {
						#  ⮣ 
						$HiddenRegularPackages += $Package
					} else {
						# ﭭ 
						$Package["Permanence"] = "$Permanence"
						#  ﭭ ⮣ 
						$HiddenPermanentPackages += $Package
					}
				}
			} else {
				#   ⮢   
				if ($IsAppend) {
					# ᪫祭  SP1
					if ($UpdateName.length -ge 9) {
						# ⡮ ⮢   
						$PackageOwners["Updates"]["$PrimaryName"] = $PackageOwners["Primary"]["$PrimaryName"]
						#  ᭮ 
						$PackageOwners["Updates"]["$PrimaryName"]["PackageStatus"] = "$PackageStatus"
						#   ᭮ 
						$PackageOwners["Updates"]["$PrimaryName"]["UpdateName"] = "$UpdateName"
						#   ᭮ 
						$PackageOwners["Updates"]["$PrimaryName"]["UpdateVers"] = "$UpdateVers"
						#   ⠬,   栬 㣨 ⮢
						ForEach ($PackageOwner in $PackageOwners["Updates"]["$PrimaryName"]["Owners"].Keys) {
							#   ⠬,    栬 㣨 ⮢ (୨ )
							ForEach ($PackageName in $PackageOwners["Updates"]["$PrimaryName"]["Owners"]["$PackageOwner"].Keys) {
								#  
								if (!$PackageOwners["Childs"]["$PackageName"]) {
									$PackageOwners["Childs"]["$PackageName"] = @{}
								}
								#  ᭮    
								$PackageOwners["Childs"]["$PackageName"]["$PrimaryName"] = "$PackageName"
								# ᮪  ⮢  ࠭ ᭮ 
								$PackageOwners["Updates"]["$PrimaryName"]["Packages"]["$PackageName"] = @{}
							}
						}
						# 砫쭠 롮ઠ ᭮ ⮢     ᯨ᪠
						if ($UpdateList -contains "$UpdateName") {
							# ᫨  ।⠢ ᪮쪨 ﬨ ⮢,
							#   ᪫祭  ⮫쪮   襩 ᨥ.
							$PackageOwners["Names"]["$UpdateName"] = Get-Compare "$UpdateVers" $PackageOwners["Names"]["$UpdateName"]
						}
					}
				}
			}
		}
		# 쭠 롮ઠ ᭮ ⮢     ᯨ᪠
		if ($PackageOwners["Names"]) {
		#   ᭮ ⠬
			ForEach ($PrimaryName in $PackageOwners["Updates"].Keys) {
				#    
				$UpdateName = $PackageOwners["Updates"]["$PrimaryName"]["UpdateName"]
				#     ᯨ᪥
				if ($PackageOwners["Names"]["$UpdateName"]) {
					# 쪮   襩 ᨥ.
					if ($IsVers) {
						# ࠢ ᨨ     ᨨ   ᯨ᪥
						if ($PackageOwners["Updates"]["$PrimaryName"]["UpdateVers"] -eq $PackageOwners["Names"]["$UpdateName"]) {
							#    짮⥫᪨ ᯨ᮪
							$PackageOwners["Custom"]["$PrimaryName"] = $PackageOwners["Updates"]["$PrimaryName"]
						}
					} else {
						#    짮⥫᪨ ᯨ᮪
						$PackageOwners["Custom"]["$PrimaryName"] = $PackageOwners["Updates"]["$PrimaryName"]
					}
				}
			}
		}
		
		# 뢮
		if ($IsOut) {
			#  ᢥ     ⠡  䠩
			if ($NormalRegularPackages) {$(ForEach ($p in $NormalRegularPackages) {new-object PSObject -Property $p}) | Format-Table "Package Identity", "|", "  Name   ", "|", "  State  ", "|", "Release Type", "|", "Install Time", "|" -AutoSize | Out-File "$Location\Packages.txt" -Width 1000 -Encoding "ASCII"}
			#  ᢥ  ﭭ    ⠡  䠩
			if ($NormalPermanentPackages) {$(ForEach ($p in $NormalPermanentPackages) {new-object PSObject -Property $p}) | Format-Table "Package Identity", "|", "  Name   ", "|", "  State  ", "|", "Release Type", "|", "Permanence", "|", "Install Time", "|" -AutoSize | Out-File -append "$Location\Packages.txt" -Width 1000 -Encoding "ASCII"}
			#  ᢥ      ⠡  䠩
			if ($HiddenRegularPackages) {$(ForEach ($p in $HiddenRegularPackages) {new-object PSObject -Property $p}) | Format-Table "Package Identity", "|", "  Name   ", "|", "  State  ", "|", "Release Type", "|", "Visibility", "|", "Install Time", "|" -AutoSize | Out-File -append "$Location\Packages.txt" -Width 1000 -Encoding "ASCII"}
			#  ᢥ   ﭭ    ⠡  䠩
			if ($HiddenPermanentPackages) {$(ForEach ($p in $HiddenPermanentPackages) {new-object PSObject -Property $p}) | Format-Table "Package Identity", "|", "  Name   ", "|", "  State  ", "|", "Release Type", "|", "Visibility", "|", "Permanence", "|", "Install Time", "|" -AutoSize | Out-File -append "$Location\Packages.txt" -Width 1000 -Encoding "ASCII"}
		} else {
			# 뢮  ⮢
			return $PackageOwners
		}
	}
	
	function Get-ExcludedUpdates ()
	{
		#  ᪫砥  ᯨ᪠
		[string[]]$ExcludedUpdates = $null
		# ஢ઠ 䠩  ᯨ᪮ ᪫砥 
		$FilePath = "$Location\~Exclude-$osver.txt"
		if ([System.IO.File]::Exists("$FilePath")) {
			# 祭 ᯨ᪠   孥 ॣ
			[string[]]$ExcludedUpdates = ForEach ($line in [System.IO.File]::ReadAllLines("$FilePath")) {
				if ($line -match "^KB[0-9]*$") {$line.ToUpper()}
			}
		}
		# ᮪ 
		if (!$ExcludedUpdates) {
			# ,   ࠢ । , ᫨ ⠭ ⥫쭮  KB3125574
			# "KB2603229", "KB2732059", "KB2773072", "KB2813347", "KB2834140", "KB2919469", "KB2970228", "KB3006137", "KB3102429"
			if ("$osver" -eq "6.1") {$ExcludedUpdates = @("KB976902", "KB2533552", "KB2603229", "KB2667402", "KB2732059", "KB2773072", "KB2813347", "KB2834140", "KB2857650", "KB2919469", "KB2970228", "KB3006137", "KB3042058", "KB3102429", "KB3123479", "KB3138612", "KB4019990", "KB4474419")}
			# ,   ࠢ । , ᫨ ⠭ ⥫쭮  KB2919355
			# "KB2932046", "KB2934018", "KB2937592", "KB2938439"
			# ,   ࠢ । , ᫨ ⠭ ⥫쭮  KB3000850
			# KB3003057, KB3014442
			if ("$osver" -eq "6.3") {$ExcludedUpdates = @("KB2932046", "KB2934018", "KB2937592", "KB2938439", "KB3003057", "KB3004545", "KB3014442", "KB3033446", "KB3162835", "KB3115224")}
		}

		return $ExcludedUpdates
	}
	
	function Get-SupersededUpdates ($UpdateInfo = $false) {
		# ।   
		if ($UpdateInfo) {
			$FilePath = "$Location\~CustomUpdates.txt"
			# ஢ઠ 㦭 䠩
			if (![System.IO.File]::Exists("$FilePath")) {
				#  㦭 䠩
				[void][System.IO.File]::Create("$FilePath").Close()
				Write-Host "You can add your updates to file: " -NoNewline; Write-Host "~CustomUpdates.txt"  -ForegroundColor Yellow
				Write-Host
				pause
			}
		}
		# 宦 ॢ 
		Write-Host Please wait ...
		# 
		$watch = [System.Diagnostics.Stopwatch]::StartNew()
		#  ⠩
		$watch.Start()
		
		# ᮪  䨪஢
		[System.Collections.Hashtable]$Deployments = @{}
		# ᮪  㬬  ⮢
		[System.Collections.Hashtable]$PackageHash = @{}
		# ᮪  
		[System.Collections.Hashtable]$TotalUpdates = @{}
		# ᮪  ⮢
		[System.Collections.Hashtable]$TotalComponents = @{}
		
		# ⢨  㧪  㧪  ॥
		[System.Collections.Hashtable]$RegAction = @{
			"Load" = {& "$env:WINDIR\system32\reg.exe" load "$RootHiveStr\COMPONENTS" "$ComponentsPath" 2> $null}
			"UnLoad" = {& "$env:WINDIR\system32\reg.exe" unload "$RootHiveStr\COMPONENTS" 2> $null}
		}

		# 㦠  ॥  ⮢
		do {
			$null = & $RegAction["Load"]
			# 뢠  "HKLM\COMPONENTS\CanonicalData\Deployments"
			$RegHiveComponents = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS")
			# 㥬 㧪  ॥ 
		} while(!$RegHiveComponents)

		# ᪫祭  㦭  ᪠ 祭
		$filter = ''
		# 뢠  "HKLM\COMPONENTS\ServicingStackVersions"
		$RegServicingStackVers = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS\ServicingStackVersions")
		#   ⥪ 㦨
		if ($RegServicingStackVers.GetValue('LastWCPVersionToAccessStore') -match "^([0-9]*\.[0-9]*\.[0-9]*)(?:\.[0-9]*)") {
			# ᭮  ⥪ ( ⢥⮣ ୮ 祭)
			$GeneralVer = $matches[1]
			#   祭  "HKLM\COMPONENTS\ServicingStackVersions"
			ForEach ($v in $RegServicingStackVers.GetValueNames()) {
				#  騥 ᨨ ⥪
				if ($v -match "^[0-9]*\.[0-9]*\.[0-9]*\.[0-9]*") {
					#  ⥪  "([\D]+|$)"  ࠭祭 ᫥ 
					$v = $matches[0] + "([\D]+|$)"
					#   ᪫祭 ⨢ ᨩ ⥬ ⮢
					if ($v -match $GeneralVer){
						if ("$filter" -eq "") {$filter += $v} else {$filter += '|' + $v}
						break
					} else {
						if ("$filter" -eq "") {$filter += $v} else {$filter += '|' + $v}
					}
				}
				
			}
		}
		# ⨢ ᨨ ⥪  
		if ("$filter" -eq '') {$filter = '^#'}
		# 뢠  "HKLM\COMPONENTS\ServicingStackVersions"
		$RegServicingStackVers.Close()

		# 祭  ⮢    ᯨ᪮ ᪫祭
		[System.Collections.Hashtable]$PackageOwners = Get-PackagesList @(Get-ExcludedUpdates) $filter -IsVers $true
						
		cls
		# 樠   蠣 ண
		$Activity  = "Searching Superseded Updates..."
		[System.Collections.Hashtable]$Tasks = @{
			"1" = 'Calculating Packages for Deployments'
			"2" = 'Calculating Components for Deployments'
			"3" = 'Calculating Data from Deployments'
			"4" = 'Calculating Versions of Components'
			"5" = 'Calculating Superseded Updates'
		}
		$TotalSteps = $Tasks.Keys.Count
		# 樠 稪
		$PreviousComplete = 0; $Complete = 0; $n = 0; $Step = 1; $Task = $Tasks["$Step"]
		
		# 뢮 ᭮ ண
		Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
		
		# 뢠  "HKLM\components\CanonicalData\Deployments"
		$RegDeployments = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("components\CanonicalData\Deployments")
		# ᫮ 䨪஢
		$TotalCount = $RegDeployments.SubKeyCount
		#   砬 (䨪ࠬ)
		ForEach ($Deployment in $RegDeployments.GetSubKeyNames()) {
			# ᪫祭  㦭  ᪠ 祭
			if ($Deployment -notmatch "$filter") {
				# 뢠  
				$r = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("components\CanonicalData\Deployments\$Deployment")
				# ᮪ 䨪஢    
				[System.Collections.Hashtable]$DeploymentPackages = @{}
				#   祭   䨪஬ ⮢
				ForEach ($ValueName in ($r.GetValueNames() -match "p!")) {
					# ᥢ ⨢ ⥬ 䨪஢  ⡮ ⮫쪮 ,    .
					if ([System.Text.Encoding]::UTF8.getString($r.GetValue("$ValueName")) -match "(\W){1,}(.*~~[0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)(?:\..*)") {
						# 祭  
						$p = $matches[2]
						#  室  ᯨ᮪
						if ($PackageOwners["Childs"]["$p"]) {
							#   ᯨ᮪
							$DeploymentPackages["$p"] = $PackageOwners["Childs"]["$p"]
						}
					}
				}
				#   ᯨ᮪
				if ($DeploymentPackages.Count) {
					#  䨪 - ᫥   
					$DeploymentHash = $Deployment.Split('_')[-1]
					#  ⠡ ᢮ 䨪
					$Deployments["$DeploymentHash"] = @{}
					#  䨪
					$Deployments["$DeploymentHash"]["Identifier"] = $Deployment
					# ᮪ ⮢   䨪
					$Deployments["$DeploymentHash"]["Packages"] = $DeploymentPackages
					# ᮪ ⮢   䨪
					[System.Collections.SortedList]$Deployments["$DeploymentHash"]["Components"] = @{}
				}
				# 뢠  
				$r.Close()
			}
			#  稪
			$n += 1
			$Complete = [math]::Round((($n * (100 / $TotalSteps)) / $TotalCount), 0) + (100 / $TotalSteps * ($Step - 1))
			if ($Complete -gt $PreviousComplete) {
				$PreviousComplete = $Complete
				Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
			}
		}
		# 뢠  "HKLM\components\CanonicalData\Deployments"
		$RegDeployments.Close()
		
		# ᪫祭  㦭  ᪠ 祭
		if (!$UpdateInfo) {
			if ("$arch" -eq "amd64") {
				if ("$filter" -eq '^#') {$filter = "^x86_|^wow64_"} else {$filter = "^x86_|^wow64_" + '|' + "$filter"}
			}
		}
		if ("$filter" -eq '') {$filter = '^#'}

		# 뢠  "HKLM\COMPONENTS\DerivedData\Components"
		$RegComponents = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS\DerivedData\Components")
		# 樠 稪
		$Complete = 0; $n = 0; $Step++; $Task = $Tasks["$Step"]
		# ᫮ ⮢
		$TotalCount = $RegComponents.SubKeyCount
		#   ⠬
		ForEach ($Component in $RegComponents.GetSubKeyNames()) {
			# ᪫祭  㦭  ᪠ 祭 (஬ KB976902)
			if ($Component -notmatch "$filter") {
				#   - ᫥   
				$Deployment = $Component.Split('_')[-1]
				# ⢥ த⥫᪮ 䨪
				if ($Deployments["$Deployment"]) {
					# ࠩ
					if ($Component -match "\.inf_") {
						# ⢥ த⥫᪮ 䨪  ⮢
						$Deployments["$Deployment"]["DeploymentComponent"] = "$Component"
						# ⢥ ⭮ 
						$Deployments["$Deployment"]["Components"]["$Component"] = "$Deployment"
					} else {
						# ⢥ த⥫᪮ 䨪  ⮢
						$Deployments["$Deployment"]["DeploymentComponent"] = "$Component"
					}
				} else {
					# 뢠  
					$r = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS\DerivedData\Components\$Component")
					#   祭 
					ForEach ($p in $r.GetValueNames()) {
						# ᫨ 祭 ᮤঠ  䨪஢
						if ($p -match "^(?:c!.*_)(.*)") {
							$Deployment = $matches[1]
							# ᫨ ᯨ᮪ 䨪஢ ᮤন  䨪
							if ($Deployments["$Deployment"]) {
								# ⢥ ⭮ 
								$Deployments["$Deployment"]["Components"]["$Component"] = "$Deployment"
							}
						}
					}
				}
				# 뢠  
				$r.Close()
			}
			#  稪
			$n += 1
			$Complete = [math]::Round((($n * (100 / $TotalSteps)) / $TotalCount), 0) + (100 / $TotalSteps * ($Step - 1))
			if ($Complete -gt $PreviousComplete) {
				$PreviousComplete = $Complete
				Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
			}
		}
		# 뢠  "HKLM\COMPONENTS\DerivedData\Components"
		$RegComponents.Close()
		
		# 樠 稪
		$Complete = 0; $n = 0; $Step++; $Task = $Tasks["$Step"]
		# ᫮ 䨪஢
		$TotalCount = $Deployments.Keys.Count
		#   䨪ࠬ
		ForEach ($Deployment in $Deployments.Keys) {
			#   ⠬    䨪
			ForEach ($p in $Deployments["$Deployment"]["Packages"].Keys) {
				#  ᭮    
				ForEach ($PrimaryPackage in $PackageOwners["Childs"]["$p"].Keys) {
					# ⡮ ᭮ ⮢    ᯨ᪥ 䨪஢
					if (!$TotalUpdates["$PrimaryPackage"]){$TotalUpdates["$PrimaryPackage"] = $PackageOwners["Updates"]["$PrimaryPackage"]}
					# ᮪[ ᭮ ⮢][][ ⮢][䨪] = ꥪ 䨪
					$TotalUpdates["$PrimaryPackage"]["Packages"]["$p"]["$Deployment"] = $Deployments["$Deployment"]
				}
			}
			#   ⠬   䨪
			ForEach ($Component in $Deployments["$Deployment"]["Components"].Keys) {
				# 祭 ୠ  
				if ($Component -match "(.*_)([0-9]*\.[0-9]*)(\.[0-9]*\.[0-9]*)(?:_)(.*_)") {
					#  
					$valPattern = $matches[1] + $matches[4]
					#   
					$verVal = $matches[2] + $matches[3]
					#  ࠧ  㯯 ⮢
					$verSection = $matches[2]
					# ᮪[][ ࠧ][][䨪] = ꥪ 䨪 
					if (!$TotalComponents["$valPattern"]){$TotalComponents["$valPattern"] = @{}}
					if (!$TotalComponents["$valPattern"]["$verSection"]){$TotalComponents["$valPattern"]["$verSection"] = @{}}
					if (!$TotalComponents["$valPattern"]["$verSection"]["$verVal"]){$TotalComponents["$valPattern"]["$verSection"]["$verVal"] = @{}}
					$TotalComponents["$valPattern"]["$verSection"]["$verVal"]["$Deployment"] = $Deployments["$Deployment"]
				}
			}
			#  稪
			$n += 1
			$Complete = [math]::Round((($n * (100 / $TotalSteps)) / $TotalCount), 0) + (100 / $TotalSteps * ($Step - 1))
			if ($Complete -gt $PreviousComplete) {
				$PreviousComplete = $Complete
				Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
			}
		}

		# 祭 ᯨ᪠ ⮢  । 㠫 ᨩ
		[System.Collections.Hashtable]$TotalVersions = @{}
		# 砥 祭   "HKLM\COMPONENTS\ServicingStackVersions"  ᢮⢠ "LastWCPVersionToAccessStore"
		$RegStackKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS\ServicingStackVersions")
		$RegStackVer = $RegStackKey.GetValue("LastWCPVersionToAccessStore")
		$RegStackKey.Close()
		# 뢠   ﬨ ⮢ "HKLM\COMPONENTS\DerivedData\VersionedIndex\$RegStackVer\ComponentFamilies"
		$RegTotalVersions = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS\DerivedData\VersionedIndex\$RegStackVer\ComponentFamilies")
		# 樠 稪
		$Complete = 0; $n = 0; $Step++; $Task = $Tasks["$Step"]
		#  ᨩ ⮢
		$TotalCount = $RegTotalVersions.SubKeyCount
		#  ᯥ譮 
		if ($RegTotalVersions) {
			#    ⮢
			ForEach ($Component in $RegTotalVersions.GetSubKeyNames()) {
				# 祭 ୠ  
				if ($Component -match "(.*_)(?:.*)$") {
					$valPattern = $matches[1]
					# ᥢ ⨢ ⥬ ⮢  ⡮ ⮫쪮 ⮢,    .
					if ($TotalComponents.ContainsKey($valPattern)){
						$TotalVersions["$valPattern"] += , "$RegWinners\$Component"
					}
				}
				#  稪
				$n += 1
				$Complete = [math]::Round((($n * (100 / $TotalSteps)) / $TotalCount), 0) + (100 / $TotalSteps * ($Step - 1))
				if ($Complete -gt $PreviousComplete) {
					$PreviousComplete = $Complete
					Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
				}
			}
			# 뢠  "HKLM\COMPONENTS\DerivedData\VersionedIndex\$RegStackVer\ComponentFamilies"
			$RegTotalVersions.Close()
		}

		# 㥬 㧪  ॥ 
		$RegHiveComponents.Close()
			
		# ᫨  㤠    ᫥  㠫 ᨩ
		if (!$TotalVersions.Count) {try {throw 'TotalVersions is empty'} catch {return}}
		
		# 樠 稪
		$Complete = 0; $n = 0; $Step++; $Task = $Tasks["$Step"]
		# ᫮ ⮢
		$TotalCount = $TotalComponents.count
		#   ⠬
		ForEach ($Component in ($TotalComponents.GetEnumerator() | Sort -Property Name)) {
			#  
			$valPattern = $Component.Name
			#    ࠧ 
			ForEach ($verSection in $TotalComponents["$valPattern"].Keys) {
				# ᮪  
				[System.Collections.Hashtable]$vUpdates = @{}
				[System.Collections.Hashtable]$uUpdates = @{}
				#   ᥬ   ࠧ 
				ForEach ($verVal in $TotalComponents["$valPattern"]["$verSection"].Keys) {
					#   䨪ࠬ   ᨨ 
					ForEach ($Deployment in $TotalComponents["$valPattern"]["$verSection"]["$verVal"].Keys) {
						#   ⠬   祭  Deployment
						ForEach ($p in $TotalComponents["$valPattern"]["$verSection"]["$verVal"]["$Deployment"]["Packages"].Keys) {
							#  ᭮ 
							ForEach ($u in $PackageOwners["Childs"]["$p"].Keys) {
								#  
								$v = $verVal
								# ᪮쪮       (Equal)
								if (!$vUpdates["$v"]) { $vUpdates["$v"] = @{} }
								$vUpdates["$v"]["$u"] = "$v"
								#     ᪮쪮 ᨩ 
								if (!$uUpdates["$u"]) { $uUpdates["$u"] = @{} }
								$uUpdates["$u"]["$v"] = "$u"
							}
						}
					}
				}
				# 㠫쭠  
				[string]$ver = "$null"
				# "Actual" - 㠫쭠   ᮢ  ᨥ 
				[string]$Condition = "Actual"
				# ᮪ 㠫  
				[System.Collections.Hashtable]$aUpdates = @{}
				# ᮪   
				[System.Collections.Hashtable]$tUpdates = $uUpdates.Clone()
				#   ஢, ᫨   ⮫쪮  
				if ($tUpdates.Count) {
					if ($tUpdates.Count -eq 1) {
						$aUpdates = $tUpdates.Clone()
					} else {
						# । 㠫쭮 ᨨ 
						if ($TotalVersions.ContainsKey("$valPattern")) {
							# ᫨ 祭   ᨢ ⮢
							$ver = Get-Version $TotalVersions["$valPattern"] $verSection
							# 室  
							$NativeVer = $ver
							# 祭 㠫  
							$aUpdates = $vUpdates["$ver"]
						}
						# 㠫   ।
						if (!$aUpdates.Count){
							# "Different" - 㠫쭠   ᮮ⢥  ⮢  ⮣ 
							# "Absent" - 㠫쭠    뫠 ।,   
							if (!$ver) {$Condition = "Absent"} else {$Condition = "Different"}
							# 롮 襩 ᨨ  
							ForEach ($v in $vUpdates.Keys) {
								$ver = Get-Compare $v $ver
							}
							# 室   ᨬ쭮 ᨨ  
							if ("$ver" -ne "$NativeVer") {$aUpdates = $vUpdates["$ver"]}
						}
						# "Equal - 㠫쭠   ᮮ⢥ ᪮쪨 ࠧ 
						if ($aUpdates.Count -gt 1) {
							$Condition = "Equal"
						}
					}
				}
				# ⢠ 
				[System.Collections.Hashtable]$ComponentProperties = @{}
				$ComponentProperties["Version"] = $ver # 
				$ComponentProperties["Component"] = $valPattern # 
				$ComponentProperties["Updates"] = $uUpdates #   
				$ComponentProperties["ActualUpdates"] = $aUpdates # 㠫  
				$ComponentProperties["Condition"] = $Condition # ﭨ 
				# 㠫  
				if ($aUpdates.Count) { 
					#   㠫 
					ForEach ($u in $aUpdates.Keys) {
						#  㠫 ⮢  ᮮ⢥    ﭨ
						$TotalUpdates["$u"]["$Condition"] += , $ComponentProperties
						# ᪫祭 㠫   ᯨ᪠    
						$tUpdates.Remove("$u")
					}
				}
				#   
				if ($tUpdates.Count){
					# 롮 ﭨ  ॢ ⮢
					if ($Condition -eq "Actual") {$Condition = "Obsolete"} else {$Condition = "Obsolete_$Condition"}
					#  ॢ ⮢  ᮮ⢥    ﭨ
					ForEach ($u in $tUpdates.Keys) {
						$TotalUpdates["$u"]["$Condition"] += , $ComponentProperties
					}
				}
			}
			#  稪
			$n += 1
			$Complete = [math]::Round((($n * (100 / $TotalSteps)) / $TotalCount), 0) + (100 / $TotalSteps * ($Step - 1))
			if ($Complete -gt $PreviousComplete) {
				$PreviousComplete = $Complete
				Write-Progress -Activity "$Activity" -Status "Step $Step of $TotalSteps | $Task" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
			}
		}
		
		# 騩 ᯨ᮪ ॢ 
		[System.Collections.Hashtable]$SupersededUpdates = $TotalUpdates.Clone()
		# 騩 ᯨ᮪ 㠫 
		[System.Collections.Hashtable]$ActualUpdates = @{}
		# ନ஢ ᯨ᪠ ॢ  㠫 
		ForEach ($u in $TotalUpdates.Keys) {
			#  ᮮ⢥⢨  稥  ᢨ 㠫 ⮢
			if (($TotalUpdates["$u"]["Actual"]) -or ($TotalUpdates["$u"]["Different"])) {
				# ᫨  ᮤন 㠫 
				$ActualUpdates["$u"] = $TotalUpdates["$u"]
				#   ᪫砥  ॢ
				$SupersededUpdates.Remove("$u")
			}
		}
		# ᮪ ᪫祭 -   ᪫  ॢ
		ForEach ($u in $PackageOwners["Custom"].Keys) {$SupersededUpdates.Remove("$u")}
		
		# ஢ઠ ॢ   稥 ⮢  ᮬ "Equal"
		[string[]]$SupersededUpdatesList =  $SupersededUpdates.Keys
		#   ॢ訬 
		ForEach ($u in $SupersededUpdatesList) {
			#    ᯨ᪥ ( 㤠)
			if ($SupersededUpdates["$u"]) {
				#      ᮬ "Equal"
				if ($SupersededUpdates["$u"]["Equal"]) {
					#   ⠬  ᮬ "Equal"
					ForEach ($Item in $SupersededUpdates["$u"]["Equal"]) {
						# ᥢ  祭
						if ($Item["ActualUpdates"].Count) {
							# ᮪ 㠫   
							[System.Collections.SortedList]$ActualEquals = @{}
							# ᫨  ᯨ᪥ 㠫     ,
							# 室饥  騩 ᯨ᮪ 㠫 ,   ய᪠
							ForEach ($e in $Item["ActualUpdates"].Keys) {
								if ($ActualUpdates.ContainsKey("$e")) {
									$ActualEquals.Clear(); Break
								} else { $ActualEquals["$e"] = "$u"	}
							}
							#  ⨢ 砥 㠫쭮    訬 浪 ஬
							#   騩 ᯨ 㠫   ᪫砥  ॢ
							if ($ActualEquals.Count) {
								#   訬 ஬  $aUpdates
								$sUpdate = $ActualEquals.GetKey($ActualEquals.Count - 1)
								# 塞   㠫
								$ActualUpdates["$sUpdate"] = $SupersededUpdates["$sUpdate"]
								# 塞   ॢ
								$SupersededUpdates.Remove("$sUpdate")
							}
						}
					}
				}
			}
		}

		# C ⮢ ॢ 
		$Conditions = "Obsolete", "Obsolete_Absent", "Obsolete_Different", "Obsolete_Equal", "Equal"
		# ᮪ ॢ 
		[string[]]$SupersededUpdatesList =  $SupersededUpdates.Keys
		# । ,   ॢ訥 
		for ($step = 1; $step -le 3; $step++) {
			#   ॢ訬 
			ForEach ($u in $SupersededUpdatesList) {
				# ᮪ ,    
				[System.Collections.SortedList]$TotalActuals = @{}
				[System.Collections.SortedList]$EveryUpdates = @{}
				[System.Collections.SortedList]$TotalReplace = @{}
				[System.Collections.SortedList]$LeastReplace = @{}
				if ($SupersededUpdates["$u"]["ActualUpdates"]) {
					$TotalReplace = $SupersededUpdates["$u"]["ActualUpdates"].Clone()
				}
				#   ᠬ ⮢
				ForEach ($Condition in $Conditions) {
					# ᫨  ᮤন    ᮬ
					if ($SupersededUpdates["$u"]["$Condition"]) {
						#    ⠬
						ForEach ($Item in $SupersededUpdates["$u"]["$Condition"]) {
							# ᥢ  祭
							if ($Item["ActualUpdates"].Count) {
								Switch($step){
									# ।  ,   ॢ訥 
									1 {
										#   㠫    
										ForEach ($e in $Item["ActualUpdates"].Keys) {
											# ஢ઠ ⮢  ᮬ "Equal"
											if ("$Condition" -match "Equal") {
												if ($ActualUpdates.ContainsKey("$e")) {
													$TotalActuals["$e"] = "$u" # 塞  ᯨ᮪ 㠫 
												}
											} else {$TotalActuals["$e"] = "$u"} # 塞  ᯨ᮪ 㠫 
										}
									}
									# । ⮣ , ஥   ॢ訥 
									2 {
										#   㠫    
										$ActualsCount  = $SupersededUpdates["$u"]["ActualUpdates"].count - 1
										for ($n = $ActualsCount; $n -ge 0; $n--) {
											# ⡮ ,     ,
											#  ४뢠   .
											$a = $SupersededUpdates["$u"]["ActualUpdates"].GetKey($n)
											if (!($Item["Updates"]).ContainsKey("$a")) {
												$TotalReplace.Remove("$a")
											}
										}
									}
									# । 쭮 ᫠ ,   ॢ訥 
									3 {
										#   㠫    
										ForEach ($a in $SupersededUpdates["$u"]["ActualUpdates"].Keys) {
											# ᫨  ᯨ᪥      ,
											# ࠭ 㠫쭮   訬 浪 ஬, 
											# 室  ஢થ ᫥饣    
											if (($Item["Updates"]).ContainsKey("$a")) {
												$LeastReplace["$a"] = "$u" # 塞  ᯨ᮪ 㠫 
												break
											}
										}
									}
								}
							}
						}
					}
				}
				#  祭  㠫 
				if ($TotalActuals.Count){
					$SupersededUpdates["$u"]["ActualUpdates"] = $TotalActuals
				}
				if ($TotalReplace.Count){
					$SupersededUpdates["$u"]["ActualUpdates"] = $TotalReplace
				}
				if ($LeastReplace.Count){
					$SupersededUpdates["$u"]["ActualUpdates"] = $LeastReplace
				}
			}
		}

		# 뢮 ண
		Write-Progress -Activity "$Activity" -Completed -Status "All done."
		
		# 㦠  ॥  ⮢
		$ErrorCount = $error.count
		$null = & $RegAction["UnLoad"]
		if ($error.count -ne $ErrorCount) {$error.RemoveRange(0, 2)}
		
		# 뢮 १⮢  ࠡ⪠ ॢ 
		if ($UpdateInfo) {
			Out-ActualComponents $ActualUpdates $TotalUpdates
		} else {
			Out-SupersededUpdates $SupersededUpdates $TotalComponents $PackageOwners
		}
	}
	
	function Out-ActualComponents ([System.Collections.Hashtable]$ActualUpdates, [System.Collections.Hashtable]$TotalUpdates) {
		# ।   
		$FilePath = "$Location\~CustomUpdates.txt"
		# ᫨ 䠩 
		if ([System.IO.File]::Exists("$FilePath")) {
			# 祭 ᯨ᪠   孥 ॣ
			[string[]]$UpdatesList = ForEach ($line in [System.IO.File]::ReadAllLines("$FilePath")) {
				if ($line -match "^KB[0-9]*$") {$line.ToUpper()}
			}
		}

		#    
		if ($UpdatesList) {
			$StructPath = "$Location" + "Structs" + '\'
			if (![System.IO.File]::Exists("$StructPath")) {[void][System.IO.Directory]::CreateDirectory("$StructPath")}
			
			# 樠   蠣 ண
			$Activity = "Creating updates structures..."
			# ᫮ 
			$TotalSteps = $UpdatesList.Count
			# 
			$Step = 0
		}
		
		#   ᭮ ⠬
		ForEach ($u in $TotalUpdates.Keys) {
			#    
			$UpdateName = $TotalUpdates["$u"]["UpdateName"]
			# ᫨     ᮤন  짮⥫᪮ ᯨ᪥
			if ($UpdatesList -contains "$UpdateName") {
				#     䠩
				[System.Text.StringBuilder]$sb = New-Object System.Text.StringBuilder
				# ⡮  ⮢  
				[System.Collections.Hashtable]$UpdateComponents = @{}
				[System.Collections.SortedList]$UpdateComponents["Patterns"] = @{}
				[System.Collections.SortedList]$UpdateComponents["Components"] = @{}
				#  
				$UpdateVers = $TotalUpdates["$u"]["UpdateVers"]
				#   䠩
				$UpdateStructure = "$StructPath" + "$UpdateName" + '_' + "$UpdateVers" + '.txt'
				#  
				[void]$sb.AppendLine("Update: $UpdateName")
				
				#  
				[void]$sb.AppendLine("Version: $UpdateVers")
				[void]$sb.AppendLine()
				
				# 뢮  ⮢   
				[void]$sb.Append("PrimaryPackage: ")
				[void]$sb.AppendLine("$u")
				[void]$sb.AppendLine()
				#   栬 ⮢
				ForEach ($Owner in $TotalUpdates["$u"]["Owners"].Keys) {
					# 
					[void]$sb.AppendLine("Owner : $Owner")
					# 
					[void]$sb.AppendLine("Packages:")
					#   ⠬  
					ForEach ($Package in $TotalUpdates["$u"]["Owners"]["$Owner"].Keys) {
						# 
						[void]$sb.AppendLine("$Package")									
					}
					[void]$sb.AppendLine()
				}
				#   ⠬  
				ForEach ($Package in $TotalUpdates["$u"]["Packages"].Keys) {
					#  ᮤন 䨪  
					if ($TotalUpdates["$u"]["Packages"]["$Package"].Count) {
						#  
						[void]$sb.AppendLine()
						[void]$sb.AppendLine("PACKAGE NAME: $Package")
						[void]$sb.AppendLine()
						#   䨪ࠬ   
						ForEach ($Deployment in $TotalUpdates["$u"]["Packages"]["$Package"].Keys) {
							# 䨪
							[void]$sb.Append("DeploymentComponent: ")
							[void]$sb.AppendLine($TotalUpdates["$u"]["Packages"]["$Package"]["$Deployment"]["DeploymentComponent"])
							#   ⠬   䨪
							[void]$sb.AppendLine("Components:")
							ForEach ($Component in $TotalUpdates["$u"]["Packages"]["$Package"]["$Deployment"]["Components"].Keys) {
								[void]$sb.AppendLine("$Component")
								# 祭 ୠ  ᫥饣 ࠢ
								if ($Component -match "(.*_)([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)(?:_)(.*_)") {
									$pattern = $matches[1] + $matches[3]
								}
								#   ᯨ᮪   
								$UpdateComponents["Patterns"]["$pattern"] += , "$Component"
								$UpdateComponents["Components"]["$Component"] = $null
							}
							[void]$sb.AppendLine()
						}
					}
				}
			
				# 樠 稪
				$PreviousComplete = 0; $Complete = 0; $n = 0; $TotalCount = 0; $Step++
				# ⡮ 㠫 ⮢  
				[void]$sb.AppendLine().AppendLine()
				#  ⮢
				$InitialConditions = "Actual", "Absent", "Different", "Equal", "Obsolete_Absent", "Obsolete_Different"
				#    ᯨ᪥
				if ($TotalUpdates["$u"]) {
					# । 饣 ᫠ ⮢
					$ActualConditions = @();
					#   ᠬ ⮢  
					ForEach ($Condition in $InitialConditions) {
						# ஢ઠ  ⮢  ࠭ 
						if ($TotalUpdates["$u"]["$Condition"]) {
							$TotalCount += $TotalUpdates["$u"]["$Condition"].Count
							$ActualConditions += , "$Condition" 
						}
					}
					#   ᠬ ⮢  
					ForEach ($Condition in $ActualConditions) {
						#  "Equal"  ॢ 
						if (($Condition -eq "Equal") -and (!$ActualUpdates["$u"])) {$Status = "Obsolete_Equal"} else {$Status = $Condition}
						# ᮪ ⮢  ࠭ 
						[System.Collections.SortedList]$CurrentComponents = @{}
						#   ⠬  ࠭ 
						ForEach ($Item in $TotalUpdates["$u"]["$Condition"]) {
							# 騩  㠫쭮 
							$CurrentComponent = $Item["Component"]
							# ୠ   ᯨ᪥ ⮢  
							if ($UpdateComponents["Patterns"].ContainsKey("$CurrentComponent")){
								#    ᯨ᮪ ⮢  ࠭ 
								ForEach ($Component in $UpdateComponents["Patterns"]["$CurrentComponent"]) {
									$CurrentComponents["$Component"] = $CurrentComponent
								}
							}
							#  稪
							$n += 1
							$Complete = [math]::Round((($n * 100) / $TotalCount), 0)
							if ($Complete -gt $PreviousComplete) {
								$PreviousComplete = $Complete
								Write-Progress -Activity "$Activity" -Status "Update $Step of $TotalSteps | $u" -CurrentOperation "$Complete% complete" -PercentComplete $Complete
							}
						}
						# ᫮ ⮢   ᮬ
						$CurrentCount = $CurrentComponents.Count
						#  ⮢
						[void]$sb.AppendLine("$Status ($CurrentCount)")
						[void]$sb.AppendLine()
						# 뢮 ᯨ᪠ 㠫 ⮢  ࠭ 
						ForEach ($ActualComponent in $CurrentComponents.Keys) {
							[void]$sb.AppendLine($ActualComponent)
							# ᪫祭 㠫쭮   饣 ᯨ᪠ 
							$UpdateComponents["Components"].Remove("$ActualComponent")
						}
						[void]$sb.AppendLine()
					}
				}
				# 뢮 ᯨ᪠  ⮢
				if ($UpdateComponents["Components"].Count) {
					# ᫮ ⮢   ᮬ
					$ObsoleteCount = $UpdateComponents["Components"].Count
					#  ⮢
					[void]$sb.AppendLine("Obsolete ($ObsoleteCount)")
					[void]$sb.AppendLine("")
					#  
					ForEach ($ObsoleteComponent in $UpdateComponents["Components"].Keys) {
						[void]$sb.AppendLine($ObsoleteComponent)
					}
					[void]$sb.AppendLine("")
				}

				# 뢮 ண   
				Write-Progress -Activity "$Activity" -Status "Update $Step of $TotalSteps | $u" -CurrentOperation "100% complete" -PercentComplete 100

				#  ଠ樨  䠩
				[System.IO.File]::WriteAllLines("$UpdateStructure", $sb, [System.Text.Encoding]::UTF8)
				Write-Host "File with structure for $UpdateName ver. $UpdateVers is created"
			}
		}
				
		# 뢮 ண
		Write-Progress -Activity "$Activity" -Completed -Status "All done."
		
		# ⠭ ⠩
		$watch.Stop()
		# ६ 믮
		Write-Host
		Write-Host Total time: $watch.Elapsed
		Write-Host
		# 㧠
		pause
	}
	
	function Out-SupersededUpdates ([System.Collections.Hashtable]$SupersededUpdates, [System.Collections.Hashtable]$TotalComponents, [System.Collections.Hashtable]$PackageOwners) {
		# 뢮 १⮢  ࠡ⪠ ॢ ⮢ 
		[System.Collections.Hashtable]$SystemUpdates = $SupersededUpdates.Clone()
		# ࠡ⪠ ᯨ᪠ ॢ ⮢ 
		if ($SystemUpdates) {
						
			#     
			[string[]]$Categories = @("Rollup Fix", ".Net Framework 4.x", ".Net Framework 3.5", "Internet Explorer", "Time Zone", "Windows Update Client", "Universal C Runtime", "Adobe Flash Player", "Servicing Stack", "Other Updates")
			# ᮪ ॢ ⮢   ⥣
			[System.Collections.Hashtable]$CategoriesUpdates = @{}
			# ᮪  ⮢   ⥣
			[System.Collections.Hashtable]$TotalCategoriesUpdates = @{}
			# 䨪樮   ⥣਩
			[System.Collections.Hashtable]$CategoriesComponents = @{"Internet Explorer" = @("_microsoft-windows-i..etexplorer-optional_31bf3856ad364e35_none_");
				"Time Zone" = @("_microsoft-windows-i..rnational-timezones_31bf3856ad364e35_none_");
				"Windows Update Client" = @("_microsoft-windows-w..wsupdateclient-core_31bf3856ad364e35_none_");
				"Universal C Runtime" = @("_microsoft-windows-ucrt_31bf3856ad364e35_none_");
				"Adobe Flash Player" = @("_adobe-flash-for-windows_31bf3856ad364e35_none_");
				"Servicing Stack" = @("_microsoft-windows-servicingstack_31bf3856ad364e35_none_")
			}
			
			# 祭 ᯨ᪠ ⥫ ⮢
			[System.Collections.SortedList]$TotalCategoriesUpdates["Rollup Fix"] = @{}
			#   ⠬
			ForEach ($u in $PackageOwners["Updates"].Keys) {
				# ஢ઠ  ᭮ 
				if ("$u" -match 'RollupFix') {
					# ᮪  ⥫ ⮢
					$TotalCategoriesUpdates["Rollup Fix"]["$u"] = $PackageOwners["Updates"]["$u"]
				}
			}
			
			# 祭 ᯨ᪠ 몮 ⮢
			[string[]]$Languages = Get-OsLangs
			# ନ஢ 蠡  ⡮ ⮢ "(?:.*~~0.0.0.0$)|(?:.*~ru-ru~0.0.0.0$)|(?:.*~en-us~0.0.0.0$)"
			$template = "(?:.*~~0.0.0.0$)"
			ForEach ($Language in $Languages) { $template += '|(?:.*~' + "$Language" + '~0.0.0.0$)' }
						
			# ᮪ ⮢  .NET Framework 4.x
			[System.Collections.SortedList]$TotalCategoriesUpdates[".Net Framework 4.x"] = @{}
			# ᮪ ⮢  .NET Framework 3.5
			[System.Collections.SortedList]$TotalCategoriesUpdates[".Net Framework 3.5"] = @{}

			# ⨥  "HKLM\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\PackageDetect"
			$RegPackageDetectKey = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackageDetect")
			#   ⠬
			ForEach ($n in $RegPackageDetectKey.GetSubKeyNames()) {
				# ࠢ ⮢  㦭 蠡
				if ("$n" -match "(NetFx[2-4])($template)") {
					# 䨪  NetFx[2-4]
					$Item = $matches[1]
					# 뢠  
					$r = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("$RegPackageDetect\$n")
					# 祭 ᯨ᪠   ﬨ  ⮣ 
					ForEach ($PackageName in $r.GetValueNames()){
						# ⡮ ᭮ ⮢  .NET Framework
						if ($PackageOwners["Childs"]["$PackageName"]) {
							#   ᭮ ⠬   
							ForEach ($p in $PackageOwners["Childs"]["$PackageName"].Keys) {
								#   ᯨ᮪ ⮢  
								if ("$Item" -eq "NetFx4") {
									#    .Net Framework 4.x
									$TotalCategoriesUpdates[".Net Framework 4.x"]["$p"] = $PackageOwners["Updates"]["$p"]
								} else {
									#    .Net Framework 3.5
									$TotalCategoriesUpdates[".Net Framework 3.5"]["$p"] = $PackageOwners["Updates"]["$p"]
								}
							}
						}
					}
					# 뢠  
					$r.Close()
				}
			}
			# 뢠 
			$RegPackageDetectKey.Close()
			
			# 祭 ᯨ᪠  ⮢    ࠭ 
			ForEach ($Title in $Categories){
				# 롮 䨪樮 
				if ($CategoriesComponents.ContainsKey("$Title")) {
					# ᮪ ⮢  
					[System.Collections.SortedList]$ComponentUpdates = @{}
					#   砬 
					ForEach ($pattern in $CategoriesComponents["$Title"]) {
						# ࠭  
						$valPattern = "$arch" + "$pattern"
						# ।  ⮢   ⮢
						if ($TotalComponents.ContainsKey("$valPattern")){
							#    ࠧ 
							ForEach ($verSection in $TotalComponents["$valPattern"].Keys) {
								#   ᥬ   ࠧ 
								ForEach ($verVal in $TotalComponents["$valPattern"]["$verSection"].Keys) {
									#   祭 Deployment   ᨨ 
									ForEach ($Deployment in $TotalComponents["$valPattern"]["$verSection"]["$verVal"].Keys) {
										#   ⠬   祭  Deployment
										ForEach ($p in $TotalComponents["$valPattern"]["$verSection"]["$verVal"]["$Deployment"]["Packages"].Keys) {
											#  ᭮    
											ForEach ($u in $PackageOwners["Childs"]["$p"].Keys) {
												#     ᯨ
												$ComponentUpdates["$u"] = $PackageOwners["Updates"]["$u"]
											}
										}
									}
								}
							}
						}
					}
					if ($ComponentUpdates.Count) {
						#  ⮢   ⥣ਨ
						$TotalCategoriesUpdates["$Title"] = $ComponentUpdates
					}
				}
			}
			cls
			# 
			[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
			# ᮪ 
			Out-Menu_Title 'Superseded Updates' $Log
			# ⠢ ᯨ᪠ ⮢   㤠  宦 㠫
			ForEach ($Title in $Categories){
				#    
				[System.Collections.SortedList]$ComponentUpdates = @{}
				if ("$Title" -eq "Other Updates"){
					# ⠫ ⥬  
					$ComponentUpdates = $SystemUpdates.Clone()
				} else {
					#     ࠭ 
					$ComponentUpdates = $TotalCategoriesUpdates["$Title"]
				}
				if ($ComponentUpdates.Count){
					# ॢ訥     
					[System.Collections.SortedList]$UpdatesList = @{}
					# 㠫    ⮣ 
					[System.Collections.SortedList]$ActualUpdates = @{}
					if ($SystemUpdates.Count) {
						# ॢ訥    ⮣ 
						ForEach ($u in $ComponentUpdates.Keys) {
							# ᫨  ५
							if ($SupersededUpdates.ContainsKey("$u")){
								# ⡮ ॢ     
								if ($SystemUpdates.ContainsKey("$u")){
									$UpdatesList["$u"] = $SupersededUpdates["$u"]
								}
							} else {
								# ⡮ 㠫    ⮣ 
								$ActualUpdates["$u"] = "$u"
							}
						}
						if ($UpdatesList.Count) {
							# ᪫祭 ॢ ⮢  ⮣   饣 ᯨ᪠
							ForEach ($u in $UpdatesList.Keys) {	$SystemUpdates.Remove("$u")}
							# 뢮 १⮢
							Out-UpdatesResults $Title $UpdatesList.Keys $ActualUpdates.Keys $SupersededUpdates $PackageOwners $Log
							
							$CategoriesUpdates["$Title"] = $UpdatesList
						}
					}
				}
			}	
			
			# ⠭ ⠩
			$watch.Stop()
			# ६ 믮
			Write-Host
			Write-Host Total time: $watch.Elapsed
			Write-Host
			# 㧠
			pause
						
			#  ⠩
			$watch.Start()
			
			cls
			# ᮪ ⮢
			Out-Menu_Title 'Superseded Packages' $Log

			# 騩 ᯨ᮪ ⮢  㤠
			[System.Collections.SortedList]$PackagesResult = @{}
			# ᮪ ⮢  
			[System.Collections.SortedList]$ComponentPackages = @{}
			# ᮪ ⮢  㤠
			[System.Collections.SortedList]$Packages = @{}
			
			# ⠢ ᯨ᪠ 㤠塞 ⮢   ⥣ਨ
			if ($SupersededUpdates) {
				ForEach ($Title in $Categories){
					#   ⥣ਨ
					if ($CategoriesUpdates["$Title"].Count) {
						# ᮪ ⮢  ⥣ਨ
						$ComponentPackages = Get-ComponentPackages $Title $CategoriesUpdates["$Title"] $Log
						# 
						if ($ComponentPackages.Count){
							#   ⠬  ⥣ਨ
							ForEach ($p in $ComponentPackages.Keys) {
								$PackagesResult["$p"] = $ComponentPackages["$p"]
							}
						}
					}
				}
			}
			Write-Host
			Write-Host "Total number of packages to remove is : " -NoNewline; Write-Host $PackagesResult.Count  -ForegroundColor Yellow
			#    㤠塞 ⮢
			if ($PackagesResult.Count) {Out-Log $Log}
			# ⠭ ⠩
			$watch.Stop()
			# ६ 믮
			Write-Host
			Write-Host Total time: $watch.Elapsed
			Write-Host
			# ஢ઠ ⬥ 믮
			if (Out-Menu_Break) {return 1}
		}
		# 롮 ᫠ 㤠塞 ⮢
		if ($PackagesResult.Count) {
			# । ᫠ ⮢  㤠
			$PackagesCount = $PackagesResult.Count
			$RemoveCount = Out-Menu_DeleteCount $PackagesCount
			#  ⮢  ᯨ᮪
			for ($n=0; $n -lt $RemoveCount; $n++) {
				$k = $PackagesResult.GetKey($n)
				$Packages["$k"] = $PackagesResult["$k"]
			}
		}

		#  ⠩
		$watch.Start()
		cls
		#  ⮢
		Out-Menu_Title 'Uninstall Packages'
		Write-Host
		#  ⮢		
		Start-DeletePackages $Packages
	
		# ⠭ ⠩
		$watch.Stop()
		# ६ 믮
		Write-Host
		Write-Host Total time: $watch.Elapsed
		Write-Host
		# 㧠
		pause
	}
	
	function Get-UpdatesByStatus ([string]$Status)
	{
		#    । ᮬ
		Write-Host
		Write-Host Get packages list. Please wait...
		Write-Host
		# 祭  ⮢  
		[System.Collections.Hashtable]$PackageOwners = Get-PackagesList -IsState $true
		# 騩 ᯨ᮪ ⮢
		[System.Collections.Hashtable]$PackagesList = $PackageOwners["Updates"]
		# ᮪ ⮢  ࠭ ﭨ
		[System.Collections.Hashtable]$StatusPackages = @{}
		#   
		ForEach ($u in $PackagesList.Keys) {
			# ⡮ ⮢  ࠭ ᮬ
			if ($PackagesList["$u"]["PackageStatus"] -eq "$Status") {
				#    ࠭ ᮬ
				$StatusPackages["$u"] = $PackagesList["$u"]
			}
		}

		cls
		# 
		[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
		# ᮪ ⮢
		Out-Menu_Title 'Packages List With Status' $Log
		# 騩 ᯨ᮪ ⮢  㤠
		[System.Collections.SortedList]$Packages = Get-ComponentPackages $Status $StatusPackages $Log
		#    㤠塞 ⮢
		if ($Packages.Count) {Out-Log $Log}
		# ஢ઠ ⬥ 믮
		if (Out-Menu_Break) {return 1}
		
		cls
		#  ⮢
		Out-Menu_Title 'Uninstall Packages'
		Write-Host
		#  ⮢
		Start-DeletePackages $Packages $Status
		# 㧠
		pause
	}
	
	function Get-UserUpdates
	{
		# ⪠   짮⥫᪮ ᯨ
		$FilePath = "$Location\~Delete.txt"
		$Title = "Custom Updates"
		# ஢ઠ 㦭 䠩
		if (![System.IO.File]::Exists("$FilePath")) {
			#  㦭 䠩
			[void][System.IO.File]::Create("$FilePath").Close()
			Write-Host "You can add your updates to file: " -NoNewline; Write-Host "~Delete.txt"  -ForegroundColor Yellow
			Write-Host
			pause
		}
		# ᫨ 䠩 
		if ([System.IO.File]::Exists("$FilePath")) {
			# 祭 ᯨ᪠   孥 ॣ
			[string[]]$UpdatesList = ForEach ($line in [System.IO.File]::ReadAllLines("$FilePath")) {
				if ($line -match "^KB[0-9]*$") {$line.ToUpper()}
			}
			cls
			# ᮪ 
			Out-Menu_Title 'Custom Updates List To Delete'
			# 뢮 ᯨ᪠ 
			Out-UpdatesResults $Title $UpdatesList
			# 㧠
			pause
			
			cls
			Write-Host
			Write-Host Get packages list. Please wait...
			Write-Host
			# 祭  ⮢  
			[System.Collections.Hashtable]$PackagesList = (Get-PackagesList $UpdatesList)["Custom"]
						
			cls
			# 
			[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
			# ᮪ ⮢
			Out-Menu_Title 'Custom Packages List To Delete' $Log
			# ᮪ ⮢  㤠
			[System.Collections.SortedList]$Packages = Get-ComponentPackages $Title $PackagesList $Log
			#    㤠塞 ⮢
			if ($Packages.Count) {Out-Log $Log}
			# ஢ઠ ⬥ 믮
			if (Out-Menu_Break) {return 1}
			
			cls
			#  ⮢
			Out-Menu_Title 'Uninstall Packages'
			Write-Host
			#  ⮢
			Start-DeletePackages $Packages
			# 㧠
			pause
		}
	}

	function Set-UserUpdatesProperty ([string]$Property)
	{
		
		$FilePath = "$Location\~CustomUpdates.txt"
		$Title = "Custom updates"
		#  䠩
		# ஢ઠ 㦭 䠩
		if (![System.IO.File]::Exists("$FilePath")) {
			#  㦭 䠩
			[void][System.IO.File]::Create("$FilePath").Close()
			Write-Host "You can add your updates to file: " -NoNewline; Write-Host "~CustomUpdates.txt"  -ForegroundColor Yellow
			Write-Host
			pause
		}
		# ᫨ 䠩 
		if ([System.IO.File]::Exists("$FilePath")) {
			# 祭 ᯨ᪠   孥 ॣ
			[string[]]$UpdatesList = ForEach ($line in [System.IO.File]::ReadAllLines("$FilePath")) {
				if ($line -match "^KB[0-9]*$") {$line.ToUpper()}
			}
			cls
			# 롮 믮塞 
			switch ("$Property") {
				'Visibility'	{ $MenuObject = 'Out-Menu_UpdateVisibility' }
				'Permanence'	{ $MenuObject = 'Out-Menu_UpdatePermanence' }
				'CurrentState'	{ $MenuObject = 'Out-Menu_UpdateState' }
			}
			# ࠭ 祭
			$PropertyValue = (& $MenuObject)
			
			cls
			# ᮪ 
			Out-Menu_Title "Custom Updates List For Change $Property"
			# 뢮 ᯨ᪠ 
			Out-UpdatesResults $Title $UpdatesList
			# 㧠
			pause
			
			cls
			Write-Host
			Write-Host Get packages list. Please wait...
			Write-Host
			# 祭  ⮢  
			[System.Collections.Hashtable]$PackagesList = (Get-PackagesList $UpdatesList)["Custom"]
			
			cls
			# 
			[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
			# ᮪ ⮢
			Out-Menu_Title "Custom Packages List For Change $Property"
			Write-Host
			# 뢮 ᯨ᪠ ⮢
			Out-PackagesResults $Title $PackagesList.Keys
			# ஢ઠ ⬥ 믮
			if (Out-Menu_Break) {return 1}
			
			cls
			#  祭 ᢮⢠ ࠭ ⮢
			Out-Menu_Title "Changing $Property value for packages" $Log
			Write-Host
			#    祭
			[System.Collections.SortedList]$SucceededPackages = $PackagesList.Clone()
			# , 祭 ᢮⢠    㤠
			[System.Collections.SortedList]$UnchangedPackages = Set-PackageProperty $PackagesList $Property $PropertyValue
			
			if ($UnchangedPackages.Count) {
				ForEach ($p in $UnchangedPackages.Keys) {$SucceededPackages.Remove("$p")}
				Write-Host
				Write-Host "Changing $Property value - is failed" -ForegroundColor Red
				Write-Host
				$UnchangedPackages.Keys | Out-Host
				Write-Host
				# 
				[void]$Log.AppendLine().AppendLine("Changing $Property value to $PropertyValue - is failed").AppendLine()
				ForEach ($p in $UnchangedPackages.Keys) {[void]$Log.AppendLine($p)}
				[void]$Log.AppendLine()
			}	
			if ($SucceededPackages.Count) {
				Write-Host
				Write-Host "Changing $Property value - is complete" -ForegroundColor Green
				Write-Host
				$SucceededPackages.Keys | Out-Host
				Write-Host
				# 
				[void]$Log.AppendLine().AppendLine("Changing $Property value to $PropertyValue - is complete").AppendLine()
				ForEach ($p in $SucceededPackages.Keys) {[void]$Log.AppendLine($p)}
				[void]$Log.AppendLine()
			}
			#  १⮢  
			Out-Log $Log
			# 㧠
			pause
		}
	}
	
	function Start-MSU_UnPack {
		#  ⥪饩 ४ਨ
		Set-Location "$Location"
		# 稥 msu 䠩
		$IsMSU = $false
		#   msu 䠩  ⥪饩 ४ਨ
		ForEach ($FilePath in [System.IO.Directory]::GetFiles("$Location", "*.msu", [System.IO.SearchOption]::TopDirectoryOnly)) {
			# msu 䠩
			$IsMSU = $true
			$FileName = [System.IO.Path]::GetFileName($FilePath)
			Write-Host "Extracting: " -NoNewline
			Write-Host "$FileName" -ForegroundColor DarkYellow
			$null = & "expand.exe" "-F:*Windows*.cab" "$FileName" "."
			Write-Host
		}
		# 稥 msu 䠩
		if ($IsMSU) {
			#   
			$SolidWidth = ($Host.UI.RawUI.WindowSize).Width - 1
			$SolidLine = "-" * $SolidWidth
			Write-Host $SolidLine
			Write-Host
		}
	}
	
	function Start-InstallPackages ([System.Collections.SortedList]$ESUPackages, [string]$Image, [string]$dism) {
		# ⠭ ⮢ ESU 
		Out-Menu_Title 'Install ESU Packages'
		# ⠭ 
		[string[]]$ProcessedPackages = $null
		#  ⮢  ᯨ᮪
		ForEach ($p in $ESUPackages.Keys) {
			$ProcessedPackages += "/PackagePath:$p "
		}
		# ꥤ  ⮢   ப
		$PackagesValue = [system.String]::Join(" ", $ProcessedPackages)
		# ⠭ ⮢
		if ($dism) {
			$DismArguments = @("/$Image", "/Add-Package $PackagesValue", "/IgnoreCheck", "/norestart")
			if ($PSCmdlet.ShouldProcess("$($ESUPackages.Count) Packages", "Installing")) {
				$ExitCode = (Start-Process -FilePath "$dism" -ArgumentList $DismArguments -Wait -NoNewWindow -PassThru).ExitCode
			}
		}
	}
	
	function Out-ESU_PackageInfo ([string]$CabName, [string]$PackageArch, [string]$version, [string]$UpdateColor, [System.Text.StringBuilder]$Log = $null) {
		#     
		if ($CabName -match "(KB[0-9]*)") {
			#  
			$UpdateName = $matches[1]
		} else {$UpdateName = $CabName}
		# 뢮 ଠ樨  ⠬   
		Write-Host "$UpdateName " -NoNewline -ForegroundColor $UpdateColor
		Write-Host "slc arch: " -NoNewline
		Write-Host "$PackageArch" -NoNewline -ForegroundColor DarkGray
		Write-Host " ver: " -NoNewline
		Write-Host "$version" -ForegroundColor DarkGray
		Write-Host
		#   
		if ($Log) {	[void]$Log.AppendLine("$UpdateName slc arch: $PackageArch ver: $version").AppendLine() }
	}
	
	function Out-Current_ImageInfo ([System.Collections.Hashtable]$CurrentWim, [bool]$IsPermanent = $false, [bool]$IsSeven = $false, [System.Text.StringBuilder]$Log = $null) {
		#  ࠧ ⨢ ⥬
		$WimPath = $CurrentWim["WimPath"]
		Write-Host "System: " -NoNewline -ForegroundColor Green
		Write-Host $WimPath -NoNewline
		# ஢ 
		$ImageIndex = $CurrentWim["ImageIndex"]
		Write-Host "	Index: " -NoNewline -ForegroundColor Green
		Write-Host $ImageIndex -NoNewline
		# ⥪ ⥬
		$ImageArch = $CurrentWim["Arch"]
		Write-Host " Arch: " -NoNewline -ForegroundColor Green
		Write-Host $ImageArch
		# । 䠩 DISM.exe
		if (!$IsPermanent) {
			# । 䠩 DISM.exe
			$dism = $CurrentWim["DismPath"]
			Write-Host "Dism: " -NoNewline -ForegroundColor Green
			Write-Host "$dism"
		}
		if (!$IsSeven) {Write-Host}
		#   
		if ($Log) {
			[void]$Log.AppendLine().AppendLine("System: $WimPath	Index: $ImageIndex Arch: $ImageArch")
			if (!$IsPermanent) {
				[void]$Log.AppendLine("Dism: $dism").AppendLine()
			} else {[void]$Log.AppendLine()}
		}
	}
	
	function Get-ESU_Component ($IsPermanent)
	{
		# 롮 ⥪饣 ࠧ ⥬  । 䠩 DISM.exe
		[System.Collections.Hashtable]$CurrentWim = Out-Menu_SelectImage
		# । ⥬ ४ਨ WinSxS
		$SystemFolder =	$CurrentWim["MountPath"] + "\Windows\WinSxS"
		# ।   䠩  ॥ COMPONENTS
		$ComponentsPath = $CurrentWim["MountPath"] + "\Windows\System32\config"
		# । ⥬ ४ਨ Manifests
		$ManifestsFolder =	$SystemFolder + "\Manifests" 
		# 窠 ஢ ᬠ
		$Image = $CurrentWim["MountPoint"]
		# । 䠩 DISM.exe
		$dism = $CurrentWim["DismPath"]
		# ⥪ ࠧ
		$ImageArch = $CurrentWim["Arch"]
		#  ॥
		$RootHive = $CurrentWim["RootHive"]
		$RootHiveStr = $CurrentWim["RootHiveStr"]
		# 䨪
		[System.Collections.Hashtable]$Deployments = @{
			# 來 x64  Windows 7
			"amd64" = "c!windowsfoundation_31bf3856ad364e35_6.1.7601.17514_615fdfe2a739474c"
			# 來 x86  Windows 7
			"x86" = "c!windowsfoundation_31bf3856ad364e35_6.1.7601.17514_0541445eeedbd616"
		}
		[System.Collections.Hashtable]$MarkTokens = @{
			# 來 x64  Windows 7
			"amd64" = "amd64_microsoft-windows-s..edsecurityupdatesai_31bf3856ad364e35_none_0e8b36cfce2fb332"
			# 來 x86  Windows 7
			"x86" = "x86_microsoft-windows-s..edsecurityupdatesai_31bf3856ad364e35_none_b26c9b4c15d241fc"
		}
		# ⥬ Windows 7
		$IsSeven = $false
		#  Windows Server 2008 R2 SP1
		if ([System.IO.File]::Exists("$ManifestsFolder\amd64_windowsserverfoundation_31bf3856ad364e35_6.1.7601.17514_none_1767904420c89fad.manifest")) {
			$Deployments["amd64"] = "c!windowsserverfoundation_31bf3856ad364e35_6.1.7601.17514_1767904420c89fad"
			$IsSeven = $true
		}
		#  ⥪ x64  Windows 7
		if ([System.IO.File]::Exists("$ManifestsFolder\amd64_windowsfoundation_31bf3856ad364e35_6.1.7601.17514_none_615fdfe2a739474c.manifest")) {
			$IsSeven = $true
		}
		#  ⥪ x86  Windows 7
		if ([System.IO.File]::Exists("$ManifestsFolder\x86_windowsfoundation_31bf3856ad364e35_6.1.7601.17514_none_0541445eeedbd616.manifest")) {
			$IsSeven = $true
		}
				
		# SLC   Extended Security Updates
		[System.Collections.SortedList]$Elements = @{}
		# SLC  64-⭮ ࠧ來
		[System.Collections.SortedList]$Elements["amd64"] = @{}
		# SLC  32-⭮ ࠧ來
		[System.Collections.SortedList]$Elements["x86"] = @{}
		
		# 
		[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
		#  ࠧ 
		$BaseVer = "6.1"
		# ⠭ ଠ⭮ 誨
		if ($IsPermanent) {
			#  
			$ElementName64 = "amd64_microsoft-windows-s..edsecurityupdatesai_31bf3856ad364e35_6.1.7603.25000_none_caceb5163345f228"
			# SLC  64-⭮ ࠧ來
			[System.Collections.Hashtable]$Elements["amd64"]["$ElementName64"] = @{
				# 
				"Version" = "6.1.7603.25000"
				#  "Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI, Culture=neutral, Version=6.1.7603.25000, publicKeyToken=31bf3856ad364e35, ProcessorArchitecture=amd64, versionScope=NonSxS"
				"Identity" = ("Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI", " Culture=neutral", " Version=6.1.7603.25000", " publicKeyToken=31bf3856ad364e35", " ProcessorArchitecture=amd64", " versionScope=NonSxS") -join(',')
				#  
				"Object" = "PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4NCjxhc3NlbWJseSB4bWxucz0idXJuOnNjaGVtYXMtbWljcm9zb2Z0LWNvbTphc20udjMiIG1hbmlmZXN0VmVyc2lvbj0iMS4wIiBjb3B5cmlnaHQ9IkNvcHlyaWdodCAoYykgTWljcm9zb2Z0IENvcnBvcmF0aW9uLiBBbGwgUmlnaHRzIFJlc2VydmVkLiI+DQogIDxhc3NlbWJseUlkZW50aXR5IG5hbWU9Ik1pY3Jvc29mdC1XaW5kb3dzLVNMQy1Db21wb25lbnQtRXh0ZW5kZWRTZWN1cml0eVVwZGF0ZXNBSSIgdmVyc2lvbj0iNi4xLjc2MDMuMjUwMDAiIHByb2Nlc3NvckFyY2hpdGVjdHVyZT0iYW1kNjQiIGxhbmd1YWdlPSJuZXV0cmFsIiBidWlsZFR5cGU9InJlbGVhc2UiIHB1YmxpY0tleVRva2VuPSIzMWJmMzg1NmFkMzY0ZTM1IiB2ZXJzaW9uU2NvcGU9Im5vblN4UyIgLz4NCjwvYXNzZW1ibHk+"
			}
			#  
			$ElementName32 = "x86_microsoft-windows-s..edsecurityupdatesai_31bf3856ad364e35_6.1.7603.25000_none_6eb019927ae880f2"
			# SLC  32-⭮ ࠧ來
			[System.Collections.Hashtable]$Elements["x86"]["$ElementName32"] = @{
				# 
				"Version" = "6.1.7603.25000"
				#  "Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI, Culture=neutral, Version=6.1.7603.25000, publicKeyToken=31bf3856ad364e35, ProcessorArchitecture=x86, versionScope=NonSxS"
				"Identity" = ("Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI", " Culture=neutral", " Version=6.1.7603.25000", " publicKeyToken=31bf3856ad364e35", " ProcessorArchitecture=x86", " versionScope=NonSxS") -join(',')
				#  
				"Object" = "PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4NCjxhc3NlbWJseSB4bWxucz0idXJuOnNjaGVtYXMtbWljcm9zb2Z0LWNvbTphc20udjMiIG1hbmlmZXN0VmVyc2lvbj0iMS4wIiBjb3B5cmlnaHQ9IkNvcHlyaWdodCAoYykgTWljcm9zb2Z0IENvcnBvcmF0aW9uLiBBbGwgUmlnaHRzIFJlc2VydmVkLiI+DQogIDxhc3NlbWJseUlkZW50aXR5IG5hbWU9Ik1pY3Jvc29mdC1XaW5kb3dzLVNMQy1Db21wb25lbnQtRXh0ZW5kZWRTZWN1cml0eVVwZGF0ZXNBSSIgdmVyc2lvbj0iNi4xLjc2MDMuMjUwMDAiIHByb2Nlc3NvckFyY2hpdGVjdHVyZT0ieDg2IiBsYW5ndWFnZT0ibmV1dHJhbCIgYnVpbGRUeXBlPSJyZWxlYXNlIiBwdWJsaWNLZXlUb2tlbj0iMzFiZjM4NTZhZDM2NGUzNSIgdmVyc2lvblNjb3BlPSJub25TeFMiIC8+DQo8L2Fzc2VtYmx5Pg=="
			}
			# ⠭ ଠ⭮ 誨
			Out-Menu_Title 'ESU Installer: Permanent fix method' $Log
			if ($Log) {[void]$Log.AppendLine()} #   
			write-host
		} else {
			# ⠭ ESU 
			Out-Menu_Title 'ESU Installer: SLC component method' $Log
			if ($Log) {[void]$Log.AppendLine()}  #   
			write-host
			# ᯠ msu 䠩
			Start-MSU_UnPack
			#  COM ꥪ
			$shell = New-Object -ComObject "Shell.Application"
			#   cab 䠩  ⥪饩 ४ਨ
			ForEach ($CabFile in $shell.NameSpace("$Location").Items()) {
				# ᮪ ⮢
				$sourceCab = @()
				#   cab 䠩
				$FilePath = $CabFile.Path
				# CAB 䠩
				if ($FilePath -match ".*\.cab$") {
					#  cab 䠩
					$CabName = [System.IO.Path]::GetFileName("$FilePath")
					# ஢ Windows    ᮤন cab 䠩
					if ($CabFile.IsBrowsable) {
						# ন CAB 䠩
						$sourceCab = $shell.NameSpace("$FilePath").Items()
					} else {
						Write-Host "$CabName " -NoNewline -ForegroundColor Yellow							
						Write-Host "is not browsable by explorer" -ForegroundColor Red
					}
				}
				# ⥪ 
				$PackageArch = "x86"
				#   ⠬ cab 䠩 
				ForEach ($item in $sourceCab) {
					#  
					if ($item.Name -match "([^_]*)(?:_.*microsoft-windows-s..edsecurityupdatesai.*_)([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)(?:.*)") {
						#  
						$version = $matches[2]
						# ⥪ 
						if (($matches[1] -eq "amd64") -or ($matches[1] -eq "wow64")) {
							$PackageArch = "amd64"
						}
						#  
						$ManifestName = $matches[0]
						#      㠫 ⮢  ⥬
						if (($IsSeven) -and ("$PackageArch" -eq "$ImageArch")) {$UpdateColor = 'Yellow'} else {$UpdateColor = 'DarkGray'}
						# 뢮 ଠ樨  ⠬   
						Out-ESU_PackageInfo $CabName $PackageArch $version $UpdateColor $Log
						#  
						$Element = [System.IO.Path]::GetFileNameWithoutExtension("$ManifestName")	
						#  SLC  ᮮ⢥饩 ࠧ來
						if (!$Elements["$PackageArch"]["$Element"]) {
							[System.Collections.Hashtable]$Elements["$PackageArch"]["$Element"] = @{}
						}
						#    Com Object
						$Elements["$PackageArch"]["$Element"]["Object"] = $item
						#   
						$Elements["$PackageArch"]["$Element"]["Version"] = "$version"
						#  "Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI, Culture=neutral, Version=6.1.7602.20587, publicKeyToken=31bf3856ad364e35, ProcessorArchitecture=amd64, versionScope=NonSxS"
						$Elements["$PackageArch"]["$Element"]["Identity"] = ("Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI", " Culture=neutral", " Version=$version", " publicKeyToken=31bf3856ad364e35", " ProcessorArchitecture=$PackageArch", " versionScope=NonSxS") -join(',')
						#  ESU 
						if (!$Elements["$PackageArch"]["$Element"]["CabNames"]) {
							[System.Collections.SortedList]$Elements["$PackageArch"]["$Element"]["CabNames"] = @{}
						}
						$Elements["$PackageArch"]["$Element"]["CabNames"]["$CabName"] = "$version"
					}
				}
			}
		}
		#  㦭 
		if (($Elements["amd64"].Count) -or ($Elements["x86"].Count)) {
			#  ࠧ ⨢ ⥬
			Out-Current_ImageInfo $CurrentWim $IsPermanent $IsSeven $Log
						
			# ⢨  㧪  㧪  ॥
			[System.Collections.Hashtable]$RegActionC = @{
				"Load" = {& "$env:WINDIR\system32\reg.exe" load "$RootHiveStr\COMPONENTS" "$ComponentsPath\COMPONENTS" 2> $null}
				"UnLoad" = {& "$env:WINDIR\system32\reg.exe" unload "$RootHiveStr\COMPONENTS" 2> $null}
			}
			# ⢨  㧪  㧪  ॥
			[System.Collections.Hashtable]$RegActionS = @{
				"Load" = {& "$env:WINDIR\system32\reg.exe" load "$RootHiveStr\SOFTWARE" "$ComponentsPath\SOFTWARE" 2> $null}
				"UnLoad" = {& "$env:WINDIR\system32\reg.exe" unload "$RootHiveStr\SOFTWARE" 2> $null}
			}
			
			$IsRemove = $false
			# ⠭  㤠 ଠ⭮ 
			if ($IsPermanent -and $IsSeven) {
				#   誨
				$ElementName = $Elements["$ImageArch"].GetKey(0)
				#    
				if ([System.IO.File]::Exists("$ManifestsFolder\$ElementName.manifest")) {
					if (!(Out-Menu_Break 'Remove permanent patch' 'Continue execute')) {
						$IsRemove = $true
					}
				}
			}
					
			# ⠭ ⮢  ⮫쪮 ࠭ 䠩
			if ($IsSeven -and ($IsRemove -or (($Elements["$ImageArch"].Count) -and (!(Out-Menu_Break 'Apply changes to system' 'Only save project'))))) {
				# 㦠  ॥  COMPONENTS
				do {
					$null = & $RegActionC["Load"]
					# 뢠  "HKLM\COMPONENTS"  "HKU\COMPONENTS"
					$RegHiveComponents = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("COMPONENTS")
					# 㥬 㧪  ॥ 
				} while(!$RegHiveComponents)
				
				# 㦠  ॥  SOFTWARE
				do {
					$null = & $RegActionS["Load"]
					# 뢠  "HKU\SOFTWARE"
					$RegHiveSoftware = [Microsoft.Win32.Registry]::$RootHive.OpenSubKey("SOFTWARE")
					# 㥬 㧪  ॥ 
				} while(!$RegHiveSoftware)
				# । ᭮  ॥
				$TargetHive = "$RootHive"
				# ।   ⮢
				$TargetFolder = "$ManifestsFolder"
				#  ॥  㬮砭
				$SectionPref = ''
				# ࠭ ࠧ來
				$ArchVals = @("$ImageArch")
			} else {
				# । ᭮  ॥
				$TargetHive = "CurrentConfig"
				# ।   ⮢
				$TargetFolder = "$Location"
				#  ॥ ESU
				$SectionPref = 'ESU\'
				#  ࠧ來
				$ArchVals = @("amd64", "x86")
			}
			
			#  Extended Security Updates
			[System.Collections.SortedList]$ESUPackages = @{}
			#  㬬
			$sha256 = New-Object -TypeName System.Security.Cryptography.SHA256CryptoServiceProvider
						
			#  祭
			if ($shell) {$DestDir = $shell.NameSpace("$TargetFolder")}
			# 室 ࠢ 㯠
			[string]$FolderSDDL = $null
			# 祭 㯠  
			if(!$IsTrustedInstaller) {
				[System.IO.DirectoryInfo]$FolderInfo = New-Object System.IO.DirectoryInfo ("$TargetFolder")
				$FolderSDDL = Set-FolderPermissions $FolderInfo
			}
			
			#   ࠧ來
			ForEach ($ArchVal in $ArchVals) {
				#   ⠬ 
				ForEach ($Element in $Elements["$ArchVal"].Keys) {
					#  
					$ManifestName = "$Element.manifest"
					# ଠ 
					if ($IsPermanent) {
						#  㤠
						if (!$IsRemove) {
							# ஢ 䠩 
							[byte[]]$OriginalFile = [System.Convert]::FromBase64String($Elements["$ArchVal"]["$Element"]["Object"])
							try {[System.IO.File]::WriteAllBytes("$TargetFolder\$ManifestName", $OriginalFile)
								Write-Host "Manifest file was created" -ForegroundColor Green
								if ($Log) {[void]$Log.AppendLine("Manifest file was created")} #   
							} catch {
								Write-Host "Something blocking file writing" -ForegroundColor Red
							}
						}
					} else {
						#    
						if (![System.IO.File]::Exists("$TargetFolder\$ManifestName")) {
							$DestDir.CopyHere($Elements["$ArchVal"]["$Element"]["Object"], 0)
						}
					}
					#    
					if ([System.IO.File]::Exists("$TargetFolder\$ManifestName")) {
						#   ⨥    "Hive\COMPONENTS\DerivedData\Components"
						$TargetSection = "$SectionPref" + "COMPONENTS\DerivedData\Components\$Element"
						# ⨥    "Hive\COMPONENTS\DerivedData\Components"
						if (($IsRemove) -and ($RegComponentKey = [Microsoft.Win32.Registry]::$TargetHive.OpenSubKey("$TargetSection", $true))) {
								#     "Hive\COMPONENTS\DerivedData\Components"
								[Microsoft.Win32.Registry]::$TargetHive.DeleteSubKey("$TargetSection")
						} else {
							#   ⨥    "Hive\COMPONENTS\DerivedData\Components"
							if ($RegComponentKey = [Microsoft.Win32.Registry]::$TargetHive.CreateSubKey("$TargetSection")) {
								#  ᢮ 䨪
								$RegComponentKey.SetValue($Deployments["$ArchVal"], (New-Object byte[](0)), "Binary")
								# ஢ ப Identity   ᨢ
								$Identity = [System.Text.Encoding]::UTF8.GetBytes($Elements["$ArchVal"]["$Element"]["Identity"])
								#  ᢮ "Identity"
								$RegComponentKey.SetValue("identity", $Identity, "Binary")
								#   㬬 䠩 
								$stream = [System.IO.File]::Open("$TargetFolder\$ManifestName",[System.IO.Filemode]::Open, [System.IO.FileAccess]::Read)
								try {
									#   㬬 cat 䠩 
									[byte[]]$S256H = $sha256.ComputeHash($stream)
								} finally {
									$stream.Close()
								}
								#  ᢮ "S256H"
								$RegComponentKey.SetValue("S256H", $S256H, "Binary")
								# 뢠 
								$RegComponentKey.Close()
							}
						}
						#   
						$version = $Elements["$ArchVal"]["$Element"]["Version"]
						#   ⨥    "Hive\SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners\MarkToken"
						$TargetSection = "$SectionPref" + "SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners\" + $MarkTokens["$ArchVal"]
						# ⨥  ᨨ   "Hive\SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners\MarkToken\BaseVer"
						if (($IsRemove) -and ($RegWinnerKey = [Microsoft.Win32.Registry]::$TargetHive.OpenSubKey("$TargetSection", $true))) {
								#  ࠧ
								if ($RegWinnerVerKey = $RegWinnerKey.OpenSubKey("$BaseVer", $true)) {
									#  ᨨ 
									$RegWinnerVerKey.DeleteValue("$version")
								}
						} else {
							#   ⨥    "Hive\SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners\MarkToken"
							if ($RegWinnerKey = [Microsoft.Win32.Registry]::$TargetHive.CreateSubKey("$TargetSection")) {
								#   ⨥  ᨨ   "Hive\SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners\MarkToken\BaseVer"
								if ($RegWinnerVerKey = $RegWinnerKey.CreateSubKey("$BaseVer")) {
									#  ᨨ 
									$RegWinnerVerKey.SetValue("$version", [byte[]](0x1), "Binary")
									# 뢠 
									$RegWinnerVerKey.Close()
								}
							}
						}
												
						# ଠ 祭
						$BaseVal = ""
						#    ࠧ
						ForEach ($subver in $RegWinnerKey.GetSubKeyNames()) {
							# ⨥  ᨨ ࠧ
							if ($RegWinnerVerKey = $RegWinnerKey.OpenSubKey("$subver", $true)) {
								# 祭 ᯨ᪠  ᨩ
								[string[]]$versions = ForEach ($val in $RegWinnerVerKey.GetValueNames()){$val.Split('',[System.StringSplitOptions]::RemoveEmptyEntries)}
								#  ࠧ  ᨨ
								if ($versions.count) {
									#  
									[string]$version = ""
									# 롮 襩 ᨨ
									ForEach ($v in $versions) {
										# 롮 襣 祭
										$version = Get-Compare $v $version
									}
									#   ᢠ 祭  㬮砭
									$RegWinnerVerKey.SetValue("", "$version", "String")
								} else {
									#  ᨨ ࠧ
									$RegWinnerKey.DeleteSubKey("$subver")
									$subver = ""
								}
							}
							#  
							if (![string]::IsNullOrEmpty($subver)) {
								# ࠢ ᨩ ࠧ
								$BaseVal = Get-Compare $subver $BaseVal
								# 뢠 
								$RegWinnerVerKey.Close()
							}
						}
						if (![string]::IsNullOrEmpty($BaseVal)) {
							#   ᢠ 祭  㬮砭
							$RegWinnerKey.SetValue("", "$BaseVal", "String")
							# 뢠 
							$RegWinnerKey.Close()
						} else {
							#   ⠢  "Hive\SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners"
							[Microsoft.Win32.Registry]::$TargetHive.DeleteSubKey("$TargetSection")
						}
						
						#  䠩 
						if ($IsRemove) {
							#    
							if ([System.IO.File]::Exists("$TargetFolder\$ManifestName")) {
								try {[System.IO.File]::Delete("$TargetFolder\$ManifestName")
									Write-Host "Manifest file was removed" -ForegroundColor Green
									if ($Log) {[void]$Log.AppendLine("Manifest file was removed")} #   
								} catch {
									Write-Host "Something blocking file removing" -ForegroundColor Red
								}
							}
						}
						
						#    㬮砭
						$ElementColor = 'DarkGray'
						# । ⮢
						if ($TargetHive -ne "CurrentConfig") {
							#  ⮢ 
							if ($Elements["$ArchVal"]["$Element"]["CabNames"]) {
								ForEach ($CabName in $Elements["$ArchVal"]["$Element"]["CabNames"].Keys) {
									$ESUPackages["$CabName"] = $Elements["$ArchVal"]["$Element"]["CabNames"]["$CabName"]
								}
							}
							#  梥     ⥬
							if (($IsSeven) -and ("$ArchVal" -eq "$ImageArch")) {$ElementColor = 'Green'}
						}
						# 뢮 ଠ樨  塞 ⠬
						Write-Host "Element " -NoNewline -ForegroundColor $ElementColor
						Write-Host "arch: " -NoNewline
						Write-Host "$ArchVal" -NoNewline -ForegroundColor DarkGray
						Write-Host " ver: " -NoNewline
						Write-Host "$version" -ForegroundColor DarkGray
						Write-Host
						if ($Log) {[void]$Log.AppendLine("Element arch: $ArchVal ver: $version").AppendLine()} #   
					}
				}
				# ࠭ 
				if ($TargetHive -eq "CurrentConfig") {
					# ࠭ 
					$Data = $null
					# 䨪  䠩
					if ("$ArchVal" -eq "amd64") {$ArchSuffix = 'x64'} else {$ArchSuffix = 'x86'}
					#   reg 䠩  ᯮ
					$file = "$Location\Offline-$ArchSuffix.reg"
					if ($RegEsuKey = [Microsoft.Win32.Registry]::CurrentConfig.OpenSubKey("ESU")) {
						# ᯮ 祩 ॥  reg 䠩
						$null = & "$env:WINDIR\system32\reg.exe" export "HKCC\ESU" "$file" "/y"
						#   ६ 祩
						[Microsoft.Win32.Registry]::CurrentConfig.DeleteSubKeyTree("ESU")
					}
					# ஢ઠ reg 䠩
					if ([System.IO.File]::Exists("$file")) {
						# ⥭   reg 䠩
						[System.IO.StreamReader]$reader = "$file"
						$Data = $null
						#  譨 ப  reg 䠩
						While (!$reader.EndOfStream) {
							$line = $reader.ReadLine()
							if ($line -match "HKEY_CURRENT_CONFIG") {
								$nextline = $reader.ReadLine()
								if ($nextline -ne '') {
									$line = $line.Replace("HKEY_CURRENT_CONFIG\ESU", "HKEY_USERS")
									$Data += ,$line
									$Data += ,$nextline
								}
							} else {$Data += ,$line}
						}
						$reader.Close()
						# ࠭   reg 䠩  ⥣樨  ࠧ
						[System.IO.File]::WriteAllLines("$file", $Data, [System.Text.Encoding]::Unicode)
					}
				}
			}
			# 뢠 , ᫨  뫨 
			if ($RegHiveComponents) {$RegHiveComponents.Close()}
			if ($RegHiveSoftware) {$RegHiveSoftware.Close()}
			# 㦠  ॥  ⮢
			$ErrorCount = $error.count
			$null = & $RegActionC["UnLoad"]
			if ($error.count -ne $ErrorCount) {$error.RemoveRange(0, 2)}
			# 㦠  ॥ SOFTWARE
			$ErrorCount = $error.count
			$null = & $RegActionS["UnLoad"]
			if ($error.count -ne $ErrorCount) {$error.RemoveRange(0, 2)}
			# 饭 室   ⬥ ࠢ 㯠
			if ($FolderSDDL) {Set-FolderPermissions $FolderInfo $FolderSDDL}
			
			#   ⠭
			if (($ESUPackages.Count) -and (!(Out-Menu_Break 'Start install packages' 'Only preserve changes'))) {
				# ⠭ ⮢
				Start-InstallPackages $ESUPackages $Image $dism
			}
			Write-Host
			Write-Host "Operations is complete" -ForegroundColor Green
			Write-Host
			if ($Log) {[void]$Log.AppendLine("Operations is complete")} #   
		} else {Write-Host "Suitable packages not found" -ForegroundColor Red}
		Write-Host
		# 뢮  
		Out-Log $Log
		pause
	}
	
	function Get-ESU_Suppression
	{
		# 롮 ⥪饣 ࠧ ⥬  । 䠩 DISM.exe
		[System.Collections.Hashtable]$CurrentWim = Out-Menu_SelectImage
		# । ⥬ ४ਨ WinSxS
		$SystemFolder =	$CurrentWim["MountPath"] + "\Windows\WinSxS"
		# 窠 ஢ ᬠ
		$Image = $CurrentWim["MountPoint"]
		# । 䠩 DISM.exe
		$dism = $CurrentWim["DismPath"]
		# ⥪ ࠧ
		$ImageArch = $CurrentWim["Arch"]
		#    䠩 suppression.xml
		$TargetComponent = $null
		# ⥬ Windows 7
		$IsSeven = $false
		#  Windows 7
		if ([System.IO.Directory]::Exists("$SystemFolder\x86_microsoft.windows.s..ation.badcomponents_31bf3856ad364e35_6.1.7600.16385_none_3868158f24725705")) {
			# ⥬ Windows 7
			$IsSeven = $true
			#  suppression.xml
			$XmlFilePath = "$SystemFolder\x86_microsoft.windows.s..ation.badcomponents_31bf3856ad364e35_6.1.7600.16385_none_3868158f24725705\suppression.xml"
		} else {$XmlFilePath = "$Location\suppression.xml"}
		# 
		[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
		# ਣ 䠩 suppression.xml
		$OriginalFile = "PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4NCjxjb21wb25lbnRzPg0KICAgIDwhLS0gVGhpcyBjb250YWlucyBhIGxpc3Qgb2YgaWRlbnRpdGllcyBvZiB0aGUgY29tcG9uZW50cyB0aGF0IHdlIHNob3VsZCBpZ25vcmUgZm9yIEFJIGVycm9ycy4gIFRoZSBlbnRyaWVzIHdpbGwgYmUgb2YgdGhlIGZvcm06DQogICAgICAgIDxjb21wb25lbnQgbmFtZT0iZm9vLCB2ZXJzaW9uPTYuMC42MDAxLjE3MDAwLCBwdWJsaWNLZXlUb2tlbj0zMWJmMzg1NmFkMzY0ZTM1LCBwcm9jZXNzb3JBcmNoaXRlY3R1cmU9eDg2LCBsYW5ndWFnZT1uZXV0cmFsLCB2ZXJzaW9uU2NvcGU9Tm9uU3hTIj48L2NvbXBvbmVudD4NCiAgICAgICAgPGNvbXBvbmVudCBuYW1lPSJiYXIsIHZlcnNpb249Ni4wLjYwMDEuMTcwMDUsIHB1YmxpY0tleVRva2VuPTMxYmYzODU2YWQzNjRlMzUsIHByb2Nlc3NvckFyY2hpdGVjdHVyZT1hbWQ2NCwgbGFuZ3VhZ2U9ZW4tdXMsIHZlcnNpb25TY29wZT1Ob25TeFMiPjwvY29tcG9uZW50Pg0KICAgIC0tPg0KDQoJPGNvbXBvbmVudCBuYW1lPSJNaWNyb3NvZnQtV2luZG93cy1QcmludGluZy1MUFJQb3J0TW9uaXRvciwgdmVyc2lvbj02LjAuNjAwMC4xNjM4NiwgcHVibGljS2V5VG9rZW49MzFiZjM4NTZhZDM2NGUzNSwgcHJvY2Vzc29yQXJjaGl0ZWN0dXJlPXg4NiwgbGFuZ3VhZ2U9bmV1dHJhbCwgdmVyc2lvblNjb3BlPU5vblN4UyI+PC9jb21wb25lbnQ+DQoJPGNvbXBvbmVudCBuYW1lPSJNaWNyb3NvZnQtV2luZG93cy1QcmludGluZy1MUFJQb3J0TW9uaXRvciwgdmVyc2lvbj02LjAuNjAwMC4xNjM4NiwgcHVibGljS2V5VG9rZW49MzFiZjM4NTZhZDM2NGUzNSwgcHJvY2Vzc29yQXJjaGl0ZWN0dXJlPWFtZDY0LCBsYW5ndWFnZT1uZXV0cmFsLCB2ZXJzaW9uU2NvcGU9Tm9uU3hTIj48L2NvbXBvbmVudD4NCgk8Y29tcG9uZW50IG5hbWU9Ik1pY3Jvc29mdC1XaW5kb3dzLVByaW50aW5nLUxQUlBvcnRNb25pdG9yLCB2ZXJzaW9uPTYuMC42MDAwLjE2Mzg2LCBwdWJsaWNLZXlUb2tlbj0zMWJmMzg1NmFkMzY0ZTM1LCBwcm9jZXNzb3JBcmNoaXRlY3R1cmU9aWE2NCwgbGFuZ3VhZ2U9bmV1dHJhbCwgdmVyc2lvblNjb3BlPU5vblN4UyI+PC9jb21wb25lbnQ+DQo8L2NvbXBvbmVudHM+DQo="
		# ⠭ ESU 
		Out-Menu_Title 'ESU Installer: Suppression method' $Log
		if ($Log) {[void]$Log.AppendLine()} #   
		write-host
		# SLC   Extended Security Updates
		[System.Collections.SortedList]$Elements = @{}
		#  Extended Security Updates
		[System.Collections.SortedList]$ESUPackages = @{}
				
		#  suppression.xml 
		if (![System.IO.File]::Exists("$XmlFilePath")) {
			# ⠭ ਣ쭮 䠩 suppression.xml
			[byte[]]$suppression = [System.Convert]::FromBase64String("$OriginalFile")
			try {[System.IO.File]::WriteAllBytes("$XmlFilePath", $suppression)
				Write-Host "Original file was restored" -ForegroundColor Green
			} catch {
				Write-Host "Something blocking file writing" -ForegroundColor Red
			}
		}
		#  suppression.xml 
		if ([System.IO.File]::Exists("$XmlFilePath")) {
			# 室 ࠢ 㯠
			[string]$FileSDDL = $null
			# 祭 䠩 ꥪ   ᢮
			[System.IO.FileInfo]$FileInfo = New-Object System.IO.FileInfo ("$XmlFilePath")
			# 祭 㯠  䠩
			if(!$IsTrustedInstaller) {
				$FileSDDL = Set-FilePermissions $FileInfo
			}
			#  ਡ ⮫쪮  ⥭ (᫨ )
			$FileInfo.IsReadOnly = $false
			# 㠫쭠 ⥬
			if ("$Image" -eq "online") {
				# ஢塞 稥 ⮢  㤠  ⠭
				if ((!$IsSeven) -or (![Microsoft.Win32.Registry]::$RootHive.OpenSubKey("SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\PackagesPending"))) {
					#  㬬 䠩 suppression.xml
					$md5 = New-Object -TypeName System.Security.Cryptography.MD5CryptoServiceProvider
					$hash = (([System.BitConverter]::ToString($md5.ComputeHash([System.IO.File]::ReadAllBytes("$XmlFilePath")))).replace('-','')).ToLower()
					# ᫨  㬬  ᮮ⢥ ਣ
					if ("$hash" -ne "773d9a9a19c31a26a532dd6739508dad") {
						if (!(Out-Menu_Break 'Restore original suppression.xml' 'Continue execute')) {
							# ⠭ ਣ쭮 䠩 suppression.xml
							[byte[]]$suppression = [System.Convert]::FromBase64String("$OriginalFile")
							try {[System.IO.File]::WriteAllBytes("$XmlFilePath", $suppression)
								Write-Host "Original file was restored" -ForegroundColor Green
							} catch {
								Write-Host "Something blocking file writing" -ForegroundColor Red
							}
							if (Out-Menu_Break) {return 1}
						}
					}
				}
			}
			
			#  ⥪饩 ४ਨ
			Set-Location "$Location"
			# ᯠ msu 䠩
			Start-MSU_UnPack
			#  COM ꥪ
			$shell = New-Object -ComObject "Shell.Application"
			#   cab 䠩  ⥪饩 ४ਨ
			ForEach ($CabFile in $shell.NameSpace("$Location").Items()) {
				# ᮪ ⮢
				$sourceCab = @()
				#   cab 䠩
				$FilePath = $CabFile.Path
				# CAB 䠩
				if ($FilePath -match ".*\.cab$") {
					#  cab 䠩
					$CabName = [System.IO.Path]::GetFileName("$FilePath")
					# ஢ Windows    ᮤন cab 䠩
					if ($CabFile.IsBrowsable) {
						# ন CAB 䠩
						$sourceCab = $shell.NameSpace("$FilePath").Items()
					} else {
						Write-Host "$CabName " -NoNewline -ForegroundColor Yellow							
						Write-Host "is not browsable by explorer" -ForegroundColor Red
					}
				}
				# ⥪ 
				$PackageArch = "x86"
				#   ⠬ cab 䠩 
				ForEach ($item in $sourceCab) {
					#  
					if ($item.Name -match "([^_]*)(?:_.*microsoft-windows-s..edsecurityupdatesai.*_)([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)(?:.*)") {
						#  
						$version = $matches[2]
						# ⥪ 
						if (($matches[1] -eq "amd64") -or ($matches[1] -eq "wow64")) {
							$PackageArch = "amd64"
						}
						#      㠫 ⮢  ⥬
						if (($IsSeven) -and ("$PackageArch" -eq "$ImageArch")) {$UpdateColor = 'Yellow'} else {$UpdateColor = 'DarkGray'}
						# 뢮 ଠ樨  ⠬   
						Out-ESU_PackageInfo $CabName $PackageArch $version $UpdateColor $Log
						#     ⨯ ⥪
						ForEach ($ElementArch in @("x86", "amd64", "ia64")) {
							#  <component name="Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI, version=6.1.7602.20587, publicKeyToken=31bf3856ad364e35, processorArchitecture=amd64, language=neutral, versionScope=NonSxS"></component>
							$Element = ("Microsoft-Windows-SLC-Component-ExtendedSecurityUpdatesAI", " version=$version", " publicKeyToken=31bf3856ad364e35", " processorArchitecture=$ElementArch", " language=neutral", " versionScope=NonSxS") -join(',')
							#  
							$Elements["$Element"] = @{
								"version" = "$version"
								"ElementArch" = "$ElementArch"
							}
						}
						#  
						if ("$PackageArch" -eq "$ImageArch") {$ESUPackages["$CabName"] = "$version"}
					}
				}
			}
			# ⥬ Windows 7   㦭   㣠 ⥬   㦭 
			if (((!$IsSeven) -and ($Elements.Count)) -or ($ESUPackages.Count)) {
				# ⥭ suppression.xml
				$xml = New-Object System.Xml.XmlDocument
				#$xml.PreserveWhitespace = $true
				$xml.Load("$XmlFilePath")
				# 騥 
				$ComponentNodes = $xml.SelectNodes("//components/component")
				#   ⠬
				ForEach ($Element in $Elements.Keys) {
					# 
					$IsAppend = $true
					#   ⠬
					ForEach ($ComponentNode in $ComponentNodes) {
						#  
						$ComponentName = $ComponentNode.name
						#  㦥   䠩
						if ("$ComponentName" -eq "$Element") {$IsAppend = $false}
					}
					$version = $Elements["$Element"]["version"]
					$ElementArch = $Elements["$Element"]["ElementArch"]
					#  
					if ($IsAppend) {	
						#  
						$XmlElement = $Xml.CreateElement("component")
						#  ਡ
						$null = ($XmlElement.SetAttribute("name", "$Element"))
						#  ⮩
						$XmlElement.InnerText = ''
						#    㬥
						if ($xml.SelectSingleNode("//components").AppendChild($XmlElement)) {
							#   㬮砭
							$ElementColor = 'DarkGray'
							#  梥     suppression.xml
							if (($IsSeven) -and ("$ElementArch" -eq "$ImageArch")) {$ElementColor = 'Green'}
							# 뢮 ଠ樨  塞 ⠬
							if ($prever -ne $version) {
								if ($prever) {Write-Host}
								Write-Host "Element " -NoNewline -ForegroundColor $ElementColor
								Write-Host "ver: " -NoNewline
								Write-Host "$version" -ForegroundColor DarkGray
								Write-Host
							}
							Write-Host "Element with arch: " -NoNewline
							Write-Host "$ElementArch" -ForegroundColor DarkGray
							#   
							if ($Log) {
								if ($prever -ne $version) {
									if ($prever) {[void]$Log.AppendLine()}
									[void]$Log.AppendLine("Element ver: $version").AppendLine()
								}
								[void]$Log.AppendLine("Element with arch: $ElementArch")
							}
							$prever = $version
						}
					}
				}
				#  㯭  
				$IsWritable = $true
				try {
					# ⮪  ࠭ 㬥  㦭 ଠ஢
					$stream = New-Object System.IO.StreamWriter("$XmlFilePath", 0, [System.Text.Encoding]::UTF8)
					# ன 㬥  ࠢ쭮 ଠ஢
					$XmlWriterSettings = New-Object System.Xml.XmlWriterSettings
					# ᯮ짮 ⠡  ஡
					$XmlWriterSettings.Indent = $true
					$XmlWriterSettings.IndentChars = "`t" #tabs instead of spaces
					$XmlWriter = [System.XML.XmlWriter]::Create($stream, $XmlWriterSettings)
					# ࠭ 㬥  ⮪]
					$Xml.WriteTo($XmlWriter)
					# ⨥ ⮪
					$XmlWriter.Close()
					# 뢠 ⮪
					$stream.Close()
				} catch {
					#   㯥  
					$IsWritable = $false
					Write-Host "Something blocking file writing" -ForegroundColor Red
				}
				#  㯥  
				if ($IsWritable) {
					#  ࠧ ⨢ ⥬
					Write-Host
					Out-Current_ImageInfo $CurrentWim $false $false $Log
					# ⠭ ⮢
					if ($IsSeven) {
						# ⠭ ⮢  ⮫쪮 ࠭ 䠩 suppression.xml
						if (($ESUPackages.Count) -and (!(Out-Menu_Break 'Start install packages' 'Only save suppression.xml'))) {
							# ⠭ ⮢
							Start-InstallPackages $ESUPackages $Image $dism
						} else {
							# ஢ઠ 䠩 suppression.xml
							if ([System.IO.File]::Exists("$XmlFilePath")) {
								# ஢ 䠩  ४ ਯ
								[System.IO.File]::Copy("$XmlFilePath", "$Location\suppression.xml", $true)
							}
						}
					}
					Write-Host
					Write-Host "Operations is complete" -ForegroundColor Green
					Write-Host
					if ($Log) {[void]$Log.AppendLine("Operations is complete")} #   
				}
			} else {Write-Host "Suitable packages not found" -ForegroundColor Red}
			# 饭 室   ⬥ ࠢ 㯠
			if ($FileSDDL) {Set-FilePermissions $FileInfo $FileSDDL}
		}
		Write-Host
		# 뢮  
		Out-Log $Log
		pause
	}
	
	function Get-SupersededState ([string]$Patche)
	{
		# ஢ઠ ﭨ 
		$IsSuperseded = $false
		$Uninstallable = (Get-ItemProperty -LiteralPath "Registry::$Patche").'Uninstallable'
		#   㤠塞
		if ("$Uninstallable" -eq "1") {
			$State = (Get-ItemProperty -LiteralPath "Registry::$Patche").'State'
			#   ॢ訬
			if ("$State" -eq "2") {
				$IsSuperseded = $true
			}
		}
			
		return $IsSuperseded
	}

	function Get-Guid ([string]$t)
	{
		# ८ࠧ  ॥ ᮮ⢥饣   ᯥ樠쭮 祭 室   㤠
		$guid = $t[7] + $t[6] + $t[5] + $t[4] + $t[3] + $t[2] + $t[1] + $t[0] + "-"
		$guid += $t[11] + $t[10] + $t[9] + $t[8] + "-"
		$guid += $t[15] + $t[14] + $t[13] + $t[12] + "-"
		$guid += $t[17] + $t[16] + $t[19] + $t[18] + "-"
		$guid += $t[21] + $t[20] + $t[23] + $t[22] + $t[25] + $t[24] + $t[27] + $t[26] + $t[29] + $t[28] + $t[31] + $t[30] 
		return $guid
	}

	function Get-OfficeUpdates
	{
		#  ॢ   
		[System.Collections.Hashtable]$ProductsNames = @{}
		[System.Collections.Hashtable]$ProductsUpdates = @{}
		[System.Collections.Hashtable]$ProductsPatches = @{}
		# ᮪  
		[string[]]$Updates = $null
		[string[]]$Patches = $null
		#    த⠬
		ForEach ($r in Get-ChildItem -Path "Registry::$RegProducts") {
			# 祭 稢騥 000000F01FEC
			if ($r.PSChildName -match "000000F01FEC") {
				$Updates = $null
				$Patches = $null
				$KeyPatches = "$r" + "\" + "Patches"
				#     ࠭ த
				ForEach ($p in Get-ChildItem -Path "Registry::$KeyPatches") {
					# ஢ઠ ﭨ 
					$IsSuperseded = Get-SupersededState $p
					#   ॢ訬
					if ($IsSuperseded) {
						# 祭  
						$Update = (Get-ItemProperty -LiteralPath "Registry::$p").'DisplayName'
						# 祭 ᯨ᪠  
						$Updates += , "$Update"
						# 祭 ᯨ᪠ 祩
						$Patches += , $p.PSChildName
					}
				}
				if ($Patches){
					# 祭  த
					$KeyProduct = "$r" + "\" + "InstallProperties"
					$ProductName = (Get-ItemProperty -LiteralPath "Registry::$KeyProduct").'DisplayName'
					$ProductsNames[$r.PSChildName] = $ProductName
					$ProductsPatches[$r.PSChildName] = $Patches
					$ProductsUpdates[$r.PSChildName] = $Updates
				}
			}
		}

		cls
		# 뢮 १⮢
		Out-Menu_Title 'MS Office Superseded Updates'
		Write-Host
		if ($ProductsNames.count -gt 0) {
			ForEach ($Key in $ProductsNames.Keys){
				#  த
				$Title = $ProductsNames["$Key"]
				# 
				$Updates = $ProductsUpdates["$Key"]
				if ($Updates) {
					# 뢮 १⮢  த
					Out-UpdatesResults $Title $Updates
				}
			}
			pause
			
			# 
			$watch = [System.Diagnostics.Stopwatch]::StartNew()
			#  ⠩
			$watch.Start()
			
			cls
			#  
			Out-Menu_Title 'Uninstall MS Office Updates'
			Write-Host
			ForEach ($Key in $ProductsNames.Keys){
				#  த
				$Title = $ProductsNames["$Key"]
				# 
				$Updates = $ProductsUpdates["$Key"]
				# 
				$Patches = $ProductsPatches["$Key"]
				# ८ࠧ   ᯥ樠쭮 祭
				$ProductGuid = Get-Guid "$Key"
				Write-Host
				Write-Host "Component: " -ForegroundColor Green -NoNewline
				Write-Host $Title -ForegroundColor Yellow
				Write-Host
				#   㤠塞 
				for ($n=0; $n -lt $Patches.count; $n++) {
					# ८ࠧ   ᯥ樠쭮 祭
					$Patche = $Patches[$n]
					$PatcheGuid = Get-Guid $Patche
					# 뢮  㤠塞 
					Write-Host "Removing: " -ForegroundColor Cyan -NoNewline
					Write-Host $Updates[$n]
					#     ᯥ樠 祭  த  
					if ((Test-Path "Registry::$RegPatches\$Patche") -and (Test-Path "Registry::$RegProducts\$Key\Patches\$Patche")) {
						# ஢ઠ ⢮ 䠩 ⠫  
						if ($PathPatche = (Get-ItemProperty -LiteralPath "Registry::$RegPatches\$Patche").'LocalPackage') {
							if (Test-Path "$PathPatche") {
								& "msiexec.exe" /package "{$ProductGuid}" /uninstall "{$PatcheGuid}" /qn /norestart | Out-Null
								if (Test-Path "Registry::$RegProducts\$Key\Patches\$Patche") {Write-Host "Error occurred" -ForegroundColor Red} else {Write-Host "Remove complete" -ForegroundColor Green}
							} else {Write-Host "Installer not exist" -ForegroundColor Magenta}
						} else {Write-Host "Path not exist" -ForegroundColor Magenta}
					} else {Write-Host "Already removed" -ForegroundColor Magenta}
				}
				#   
				$SolidWidth = ($Host.UI.RawUI.WindowSize).Width - 1
				$SolidLine = "-" * $SolidWidth
				Write-Host
				Write-Host $SolidLine
			}
			
			# ⠭ ⠩
			$watch.Stop()
			# ६ 믮
			Write-Host
			Write-Host Total time: $watch.Elapsed
			Write-Host
		} else {Write-Host "There are no updates to remove"; Write-Host}
		pause
	}
	
	function Out-Menu_Break ([string]$Accept = 'Continue execute', [string]$Decline = 'Return to Menu'){
		# 㭪 롮 - த 믮 ਯ    
		do
		{
			Write-Host
			Write-Host " [Y] " -ForegroundColor Cyan -NoNewline; Write-Host "= $Accept" -NoNewline;
			Write-Host " [*] " -ForegroundColor Cyan -NoNewline; Write-Host "= $Decline"
			Write-Host
			$KeyPress = [System.Console]::ReadKey($true)
			Switch($choice){
				{$KeyPress.Key -eq 'Y'} {return $false}
				default {return $true}
			}
		}
		until ($false)
	}
	
	function Out-Menu_Title ([string]$Title = $null, [System.Text.StringBuilder]$Log = $null){
		# 뢮 ࠬ  
		if ($Title){
			#  
			$TitleWidth = $Title.Length
			#   
			$SolidWidth = ($Host.UI.RawUI.WindowSize).Width - 1
			#   ࠭
			$BorderWidth = $SolidWidth - 2
			#  ᫥
			$LeftWidth = [math]::Floor(($BorderWidth - $TitleWidth) / 2)
			#  ࠢ
			$RightWidth = $BorderWidth - ($TitleWidth + $LeftWidth)

			#  
			$SolidLine = "*" * $SolidWidth
			#  ࠭
			$BorderLine = "*" + (" " * $BorderWidth) + "*"
			#  ࠭ ࠢ
			$RightLine = "*" + (" " * $RightWidth)
			#  ࠭ ᫥
			$LeftLine = (" " * $LeftWidth) + "*"
			#   
			$HT = "*" + (" " * $RightWidth) + $Title + (" " * $LeftWidth) + "*"
			
			# 뢮 ࠬ
			Write-Host
			$SolidLine | Out-Host
			$BorderLine | Out-Host
			$BorderLine | Out-Host
			Write-Host $RightLine -NoNewline
			Write-Host $Title -ForegroundColor White -NoNewline
			Write-Host $LeftLine
			$BorderLine | Out-Host
			$BorderLine | Out-Host
			$SolidLine | Out-Host
			
			# 뢮 ࠬ  
			if ($Log) {
				[void]$Log.AppendLine()
				[void]$Log.AppendLine($SolidLine)
				[void]$Log.AppendLine($BorderLine)
				[void]$Log.AppendLine($BorderLine)
				[void]$Log.Append($RightLine)
				[void]$Log.Append($Title)
				[void]$Log.AppendLine($LeftLine)
				[void]$Log.AppendLine($BorderLine)
				[void]$Log.AppendLine($BorderLine)
				[void]$Log.AppendLine($SolidLine)
			}
		}
	}
	
	function Out-Menu_Action ([System.Collections.Hashtable]$MenuNumbers, [System.Collections.Hashtable]$MenuActions, [string]$Title = $null, [string]$NoInput = $null, [string]$NoInputAction = $null, [string]$Color = 'Cyan')
	{
		# 뢮 
		do
		{
			#  䠩
			$date = Get-Date -format "yyyyMMddhhmmss"
			$LogFile = "$Location" + "Logs" + "\Log-" + "$date" + ".txt"
			
			cls
			# 뢮 
			if ($Title) {
				# 뢮 
				Out-Menu_Title $Title
			}	
			Write-Host
	
			# ᫮ 㭪⮢ 
			$ItemsCount = $MenuNumbers.count
			# 冷 뢮 () - 㭪 
			For ($n=0; $n -lt $ItemsCount; $n++) {
				$Num = $n+1
				Write-Host " [$Num] " -ForegroundColor "$Color" -NoNewline; Write-Host "=" $MenuNumbers["$Num"]
			}
			Write-Host
			# 뢮 ᠭ ⢨  ⢨ 롮
			if ($NoInput) {Write-Host " [No Input] " -ForegroundColor "$Color" -NoNewline; Write-Host "= $NoInput";	Write-Host}
			$choice = Read-Host "Your choice "
			cls
			# ஢ ࠢ쭮 롮 (ᮤন    ⠡), ⥬ 믮 ⢨ ᮮ⢥饥  㭪  (祭  ⠡  ࠬ).
			if ($MenuActions) {
				Switch($choice){
					{$MenuNumbers.ContainsKey("$choice")}{& $MenuActions[$MenuNumbers["$choice"]]}
					"$null" {if ($MenuActions["NoInput"]) {& $MenuActions["NoInput"]}}
					default {if ($MenuActions["default"]) {& $MenuActions["default"]}}
				}
			} else {
				Switch($choice){
					{$MenuNumbers.ContainsKey("$choice")}{return $MenuNumbers["$choice"]}
					"$null" {if ($NoInputAction) {& $NoInputAction}}
					default {Write-Host "Wrong choice, try again." -ForegroundColor Red; pause}
				}
			}
			# 뢮 訡   ண ஬
			if ($error.count) {
				# 
				[System.Text.StringBuilder]$Log = New-Object System.Text.StringBuilder
				[void]$Log.AppendLine()
				[void]$Log.AppendLine('List of received errors')
				[void]$Log.AppendLine()
				ForEach ($e in $error) {
					[void]$Log.Append($e.InvocationInfo.MyCommand).Append(" : ")
					[void]$Log.AppendLine($e.Exception.Message)
					[void]$Log.AppendLine($e.InvocationInfo.PositionMessage)
					[void]$Log.AppendLine()
				}
				Out-Log $Log
				# ᮫
				Write-Progress " " " " -completed; cls; Write-Host 'List of received errors' -ForegroundColor Red; Write-Host; $error | Out-Host
				$error.clear()
				pause
			}
		}
		until ($False)
	}
	
	function Out-Menu_Main
	{
		#  
		$Title = "Additional Windows Update Tools"
		
		#  ⠡: 㭪  - ⢨
		[System.Collections.Hashtable]$MenuActions = 
		@{
			"Get Packages List" = {Get-PackagesList -IsOut $true}
			"Superseded Updates" = {Get-SupersededUpdates}
			"Microsoft Office" = {Get-OfficeUpdates}
			"Not Present" = {Get-UpdatesByStatus "Not Present"}
			"Staged" = {Get-UpdatesByStatus "Staged"}
			"ESU Installer" = {Out-Menu_ESU_Istaller}
			"Advanced Options" = {Out-Menu_Advanced}
			"NoInput" = {Write-Host "Exit"; exit}
			"default" = {Write-Host "Wrong choice, try again." -ForegroundColor Red; pause}
		}

		# 㭪 
		[string[]]$MenuItems = @("Get Packages List", "Superseded Updates", "Microsoft Office", "Not Present", "Staged", "ESU Installer", "Advanced Options")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
		
		# 뢮  
		Out-Menu_Action $MenuNumbers $MenuActions $Title 'Exit'
	}
	
	function Out-Menu_ESU_Istaller {
		#  
		$Title = "ESU Istaller"
		
		#  ⠡: 㭪  - ⢨
		[System.Collections.Hashtable]$MenuActions = 
		@{
			"Suppression method" = {Get-ESU_Suppression}
			"SLC component method" = {Get-ESU_Component $false}
			"Permanent fix method" = {Get-ESU_Component $true}
			"NoInput" = {Out-Menu_Main}
			"default" = {Write-Host "Wrong choice, try again." -ForegroundColor Red; pause}
		}

		# 㭪 
		[string[]]$MenuItems = @("Suppression method", "SLC component method", "Permanent fix method")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
		
		# 뢮 ⥫쭮 
		Out-Menu_Action $MenuNumbers $MenuActions $Title 'Return'
	}
	
	function Out-Menu_Advanced
	{
		#  
		$Title = "Additional Windows Update Tools (Advanced)"
		
		#  ⠡: 㭪  - ⢨
		[System.Collections.Hashtable]$MenuActions = 
		@{
			"Actual Components and Structure" = {Get-SupersededUpdates $true}
			"Change Custom Updates Property" = {Out-Menu_UpdateProperty}
			"Delete Custom Updates" = {Get-UserUpdates}
			"NoInput" = {Out-Menu_Main}
			"default" = {Write-Host "Wrong choice, try again." -ForegroundColor Red; pause}
		}

		# 㭪 
		[string[]]$MenuItems = @("Actual Components and Structure", "Change Custom Updates Property", "Delete Custom Updates")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
		
		# 뢮 ⥫쭮 
		Out-Menu_Action $MenuNumbers $MenuActions $Title 'Return'
	}
	
	function Out-Menu_UpdateProperty {
		#  
		$Title = "Select Update Property"
		
		#  ⠡: 㭪  - ⢨
		[System.Collections.Hashtable]$MenuActions = 
		@{
			"Visibility" = {Set-UserUpdatesProperty 'Visibility'}
			"Permanence" = {Set-UserUpdatesProperty 'Permanence'}
			"CurrentState" = {Set-UserUpdatesProperty 'CurrentState'}
			"NoInput" = {Out-Menu_Advanced}
			"default" = {Write-Host "Wrong choice, try again." -ForegroundColor Red; pause}
		}

		# 㭪 
		[string[]]$MenuItems = @("Visibility", "Permanence", "CurrentState")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
		
		# 뢮 ⥫쭮 
		Out-Menu_Action $MenuNumbers $MenuActions $Title 'Return'
	}
	
	function Out-Menu_UpdateVisibility {
		#  
		$Title = "Select Visibility State"
		
		# 㭪 
		[string[]]$MenuItems = @("Visible", "Not Visible")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
				
		# 뢮  롮
		$Visibility = Out-Menu_Action $MenuNumbers $null $Title 'Return' 'Out-Menu_Advanced'
		#  祭
		if ("$Visibility" -eq 'Visible') {return 1} else {return 2}
	}
	
	function Out-Menu_UpdatePermanence {
		#  
		$Title = "Select Permanence State"
		
		# 㭪 
		[string[]]$MenuItems = @("Removable", "Permanent")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
				
		# 뢮  롮
		$Permanence = Out-Menu_Action $MenuNumbers $null $Title 'Return' 'Out-Menu_Advanced'
		#  祭
		if ("$Permanence" -eq 'Permanent') {return 131200} else {return 131184}
	}
	
	function Out-Menu_UpdateState {
		#  
		$Title = "Select New Status"
		
		# 㭪 
		[string[]]$MenuItems = @("Installed", "Install Pending", "Superseded", "Staged", "Not Present", "Uninstall Pending")
		# ᫮ 㭪⮢ 
		$ItemsCount = $MenuItems.count
		
		#  ⠡: 冷 뢮 () - 㭪 
		[System.Collections.Hashtable]$MenuNumbers = @{}
		For ($n=0; $n -lt $ItemsCount; $n++) {
			$Num = $n+1
			$MenuNumbers["$Num"] = $MenuItems[$n]
		}
				
		# 뢮  롮
		$Status = Out-Menu_Action $MenuNumbers $null $Title 'Return' 'Out-Menu_Advanced'
		#   
		$StatusCode = Get-StateCode $Status
		#  祭
		if ($StatusCode) {return $StatusCode}
	}
	
	function Out-Menu_DeleteCount ([int]$PackagesCount) {
		# ᫮ ⮢  㤠
		$RemoveCount = $PackagesCount
		# ᫨ ⮢  । ᫠,  짮⥫ ।⠢ 롮
		if ($PackagesCount -gt 50) {
		
			#  
			$Title = 'Specify the number of packages to be removed'
			
			# 㭪 
			[int[]]$MenuItems = @(50, 100, 150, 200, 250, 300, 350, $PackagesCount)
			# ᫮ 㭪⮢ 
			$ItemsCount = $MenuItems.count
			#  ⠡: 冷 뢮 () - 㭪 
			[System.Collections.Hashtable]$MenuNumbers = @{}
			For ($n=0; $n -lt $ItemsCount; $n++) {
				$Num = $n+1
				# 祭  饣 ᫠ ⮢
				if ($MenuItems[$n] -lt $PackagesCount) {
					#   祭
					$MenuNumbers["$Num"] = $MenuItems[$n]
					# ᫥ 祭 - 饥 ᫮ ⮢
				} else {$MenuNumbers["$Num"] = $MenuItems[$ItemsCount-1]; break}
			}
					
			# 뢮  롮
			$RemoveCount = Out-Menu_Action $MenuNumbers $null $Title 'Return' 'Out-Menu_Main'
			
			# ࠭ 祭  饣 ᫠ ⮢
			if ($RemoveCount -gt $PackagesCount) {$RemoveCount = $PackagesCount}
		}
		
		#  祭
		return $RemoveCount
	}
	
	function Out-Menu_SelectImage {
		# ᮪ ᬮ஢ ࠧ
		[System.Collections.SortedList]$MountedWims = @{}
		# ࠭ ࠧ
		[System.Collections.Hashtable]$CurrentWim = @{}
		# ஢ઠ   ⥬, ᬮ஢ ࠧ (ᮢ wimserv.exe)
		if ($WimServArray = Get-WmiObject -Class win32_process | where-object {$_.Name -eq 'wimserv.exe'}) {
			# ७祭 ⥬ ६
			if ($RegMountedImages = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey("SOFTWARE\Microsoft\wimmount\Mounted Images")) {
				#   䨪ࠬ ᬮ஢ ࠧ
				ForEach ($WimId in $RegMountedImages.GetSubKeyNames()) {
					#   ᠬ wimserv.exe
					ForEach ($WimServ in $WimServArray) {
						# 䨪 ࠧ   ᮮ⢥ ⠪  ॥
						if ($wimserv.CommandLine -eq "wimserv.exe $WimId") {
							# । 䠩 DISM.exe,  㦨 ࠧ
							$DismPath = [System.IO.Path]::GetDirectoryName($wimserv.Path) + "\" + "dism.exe"
							if ([System.IO.File]::Exists("$DismPath")) {
								# ⨥   ᢮ᢠ ᬮ஢ ࠧ
								if ($RegWimId = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey("SOFTWARE\Microsoft\wimmount\Mounted Images\$WimId")) {
									[System.Collections.Hashtable]$MountedWims["$WimId"] = @{}
									# ४ ஢
									$ImagePath = $RegWimId.GetValue('Mount Path')
									# ⥪ ࠧ
									$MountedWims["$WimId"]["Arch"] = "x86"
									if ([System.IO.Directory]::Exists("$ImagePath\Windows\SysWOW64")) {
										$MountedWims["$WimId"]["Arch"] = "amd64"
									}
									$MountedWims["$WimId"]["MountPath"] = $ImagePath
									# 窠 ஢ ᬠ
									$MountedWims["$WimId"]["MountPoint"] = "Image:$ImagePath"
									#  ࠧ 
									$MountedWims["$WimId"]["WimPath"] = $RegWimId.GetValue('WIM Path')
									# ஢ 
									$MountedWims["$WimId"]["ImageIndex"] = $RegWimId.GetValue('Image Index')
									# DISM.exe,  㦨 ࠧ
									$MountedWims["$WimId"]["DismPath"] = $DismPath
									#  ॥  祭
									$MountedWims["$WimId"]["RootHive"] = 'Users'
									# ப 祭 ࠧ ॥
									$MountedWims["$WimId"]["RootHiveStr"] = 'HKU'
								}
							}
						}
					}
				}
			}
		}
		#  ᬮ஢ ࠧ
		if ($MountedWims.Count) {
			# ஢ ⮫쪮  ࠧ
			if ($MountedWims.Count -eq 1) {
				$WimId = $MountedWims.GetKey(0)
				$CurrentWim = $MountedWims["$WimId"]
			} else {
				#  
				$Title = "Select Current Image"
				
				# ᫮ 㭪⮢ 
				$ItemsCount = $MountedWims.count
				
				#  ⠡: 冷 뢮 () - 㭪 
				[System.Collections.Hashtable]$MenuNumbers = @{}
				For ($n=0; $n -lt $ItemsCount; $n++) {
					$Num = $n+1
					$WimId = $MountedWims.GetKey($n)
					$MenuNumbers["$Num"] = "WIM: " + $MountedWims["$WimId"]["WimPath"] + " Index: " + $MountedWims["$WimId"]["ImageIndex"]
				}
						
				# 뢮  롮
				$ImageID = Out-Menu_Action $MenuNumbers $null $Title
				# ⭮ ८ࠧ 祭 
				For ($n=0; $n -lt $ItemsCount; $n++) {
					$Num = $n+1
					if ($MenuNumbers["$Num"] -eq "$ImageID") {
						$WimId = $MountedWims.GetKey($n)
						$CurrentWim = $MountedWims["$WimId"]
					}
				}
			}
		}
		# ⨢ ⥬
		if (!$CurrentWim.Count) {
			# ४ ⨢ ⥬
			$CurrentWim["MountPath"] = "$env:systemdrive"
			# 窠 ஢ ᬠ
			$CurrentWim["MountPoint"] = "online"
			#  ࠧ ⨢ ⥬
			$CurrentWim["WimPath"] = 'online'
			# ஢ 
			$CurrentWim["ImageIndex"] = 0
			# ⥪ ࠧ
			$CurrentWim["Arch"] = Get-OsArch
			# DISM.exe,  㦨 ࠧ
			$CurrentWim["DismPath"] = Get-OsDism
			#  ॥  祭
			$CurrentWim["RootHive"] = 'LocalMachine'
			# ப 祭 ࠧ ॥
			$CurrentWim["RootHiveStr"] = 'HKLM'
		}
		
		return $CurrentWim
	}
			
	function Main {
		#  ६
		New-Variable -Name RootHiveStr -Value 'HKLM' -Scope local
		New-Variable -Name RootHive -Value 'LocalMachine' -Scope local
		# 
		New-Variable -Name RegProducts -Value "HKLM\SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Products" -Scope local
		New-Variable -Name RegPatches -Value "HKLM\SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Patches" -Scope local
		# Windows
		New-Variable -Name RegWinners -Value "SOFTWARE\Microsoft\Windows\CurrentVersion\SideBySide\Winners" -Scope local
		New-Variable -Name RegPackageDetect -Value "SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\PackageDetect" -Scope local
		New-Variable -Name RegPackages -Value "SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\Packages" -Scope local
		New-Variable -Name RegPackageIndex -Value "SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\PackageIndex" -Scope local
		# । ⥬ ४ਨ  室묨 ⠬
		New-Variable -Name PackagesFolder -Value "$env:systemroot\servicing\Packages" -Scope local
		#  ॥ ⮢
		New-Variable -Name ComponentsPath -Value "$env:WINDIR\system32\config\COMPONENTS" -Scope local
		#  ⥬
		New-Variable -Name osver -Value $null -Scope local
		# ⥪ ⥬
		New-Variable -Name arch -Value $null -Scope local
		#  ४ ਯ
		New-Variable -Name Location -Value $null -Scope local
		# 
		New-Variable -Name LogFile -Value $null -Scope local
		# ਯ 饭  ࠢ TrustedInstaller
		New-Variable -Name IsTrustedInstaller -Value $false -Scope local
		# ஢ઠ
		if ([System.Console]::Title -match 'TrustedInstaller') {$IsTrustedInstaller = $true}
		
		# । ᨨ ⥬
		$SupportedOS = @("6.1", "6.2", "6.3", "10.0")
		$osver = Get-OsVers
		$IsExcluded = $SupportedOS -contains "$osver"
		#  ন ⥬
		if ("$IsExcluded" -eq "False") {
			Write-Host
			Write-Host This OS is not supported -ForegroundColor Red
			Write-Host
			pause
			# 室
			exit
		}
		# । ⥪ ⥬
		$arch = Get-OsArch
		# । ⥪饩 ४ਨ ਯ
		$Location = $(Get-CurrentLocation)["Location"]
		#  ண 
		# $ProgressPreference = 'SilentlyContinue'
		# 뢮  
		Out-Menu_Main
		pause
	}
	#  ਯ  ࠢ TrustedInstaller
	RunAsTrustedInstaller
	# ᭮ 㭪
	Main
}
UpdateTools